SEXP_CONV_PATH "Goba"

open VecOps

let firing_period = 0.2
let mine_period = 1.0
let max_speed = 150.0
let max_reverse_speed = -.50.0
let opposition = 1.0 (* wind force *)
let friction = 0.60

let bitmap_offset = (-20.0, -20.0)
let bitmap_size = (40.0, 40.0)

let max_hp = 100
let max_ammo = 50
let max_mines = 4

let ident_font = lazy (Sdlttf.open_font "font.ttf" 20)

let frames color = 
  Array.init 36
    (fun n ->
       let _, bm = Bitmaps.bitmap (`Colored (`Bitmap (Printf.sprintf "tank-%04d" (n + 1)),
					     color)) in
	 (bm,
	  Bitmaps.collision_mask_of_surface2 (
	    let _, s = Bitmaps.bitmap (`Bitmap (Printf.sprintf "tank_mask-%04d" (n + 1))) in
	      (*	     if (n = 0) then
			     begin
			     let si = Sdlvideo.surface_info s in
			     let module C = CollisionMask in
			     for y = 0 to si.Sdlvideo.h - 1 do
			     for x = 0 to si.Sdlvideo.w - 1 do
			     Printf.printf "%c" (if Int32.logand (Sdlvideo.get_pixel s x y) 0x00ffffffl <> 0l then '1' else '0')
			     done;
			     Printf.printf "\n"
			     done;
			     Printf.printf "%!";
			     end;*)
	      s
	  )
	 )	
    )

let collision_sound =
  let sample = Sfx.Sample "collision" in
  let channel = Sfx.alloc_channel () in
    fun (world : Object.world) ->
      let ch = world#channel channel in
	if not ch#playing then ch#play sample

type init = { init_at : float * float;
	      init_color : int * int * int;
	      init_direction : float }
with sexp

class t id init0 =
  (*let bitmap = lazy (Bitmaps.bitmap (`Block (11, 18, constr.Constr.tank_color), 255)) in*)
  (*let cmask = lazy (Bitmaps.collision_mask_of_surface (snd (Lazy.force bitmap))) in*)

  let init = Object.default_with init0 init_of_sexp in

object (self)
  inherit Object.t id as super

  val frames = lazy (frames init.init_color)

  val mutable at = init.init_at
  val mutable motor_speed = 0.0
  val mutable speed = (0.0, 0.0)
  val mutable firing_time = firing_period		(* how long since the last time we fired? *)
  val mutable mine_time = mine_period (* how long since we last dropped a mine? *)

  val mutable ident_surface = None
    
  val mutable hp = max_hp
  val mutable ammo = max_ammo
  val mutable mines = max_mines
  val mutable direction = 0.0

  val mutable damage_counter = 0.0

  method get_state delay =
    Object.sexp_of_state sexp_of_init
      { Object.state_super = super#get_state delay;
	state_init = { init with init_direction = direction } }

  method set_state state delay =
    let state = Object.state_of_sexp init_of_sexp state in
      super#set_state state.Object.state_super delay;
      direction <- state.Object.state_init.init_direction;

  method get_direction = direction

  method z_location = 0.0

  method set_ident str = 
    let surface = Sdlttf.render_text_blended (Lazy.force ident_font) str (128, 255, 128) in
      ident_surface <- Some surface

  method set_location at' = at <- at'
  method set_speed speed' = 
    speed <- speed';
    (* not quite correct *)
(*    if Vector.abs2 speed > 0.01 then
      direction <- atan2 (snd speed) (fst speed)*)

  method turn adjust =
    direction <- direction +. adjust;
    self#change

  method cur_frame =
    (Lazy.force frames).(
      let deg = (((pi2 -. direction) /. pi2) *. 360.0) in
      let n = int_of_float (deg /. 10.0 +. 0.5) in
      let n = n + 9 in
      let n = if n < 0 then n + 36 else n in
	(*Printf.printf "%f %f %d\n%!" direction deg n;*)
	n mod 36
    )

  method get_damage_counter = damage_counter

  method location = at
  method speed = speed
  method render (gfx:#Gfx.t) flags at = 
(*    let ofs, surface = Lazy.force bitmap in*)
(*      gfx#blit (at +| Vector.int2 bitmap_offset +| ofs) surface;*)
    gfx#blit (at +| Vector.int2 bitmap_offset) (fst self#cur_frame);
    if List.exists 
      (function 
       | `Labels -> true
       | `LabelsExcept oids when not (List.exists ((=) self#get_id) oids) -> true
       | _ -> false
      )
      flags then 
      match ident_surface with
	| None -> ()
	| Some surface ->
(*	    let size = Gfx.surface_dims surface in*)
(* doesn't work ??? *)
(*	    let pos = (at -| size /| (2, 2) +| ofs) in*)
	    let pos = (at +| Vector.int2 bitmap_offset +| (13, -15)) in
(*	      Printf.printf "size = %d,%d\n%!" (fst size) (snd size);
	      Printf.printf "pos = %d,%d\n%!" (fst pos) (snd pos);*)
	      gfx#blit pos surface

  method get_hp = hp

  method accelerate adjust = 
    motor_speed <- max max_reverse_speed (min max_speed (motor_speed +. adjust));
    self#change

  method turret_offset (angle : float) =
    bitmap_offset +|. bitmap_size /|. Vector.v2 2.0

  method private adjust_location adj = at <- at +|. adj

  method private stop_axis vec = 
    let axis = Vector.unit vec in
    let stopping_power = 
      Vector.dot (Vector.v2 motor_speed *|. (cos direction, sin direction)) axis in
      (*Printf.printf "Stopping power: %f\n%!" stopping_power;*)
      motor_speed <- motor_speed +. 0.5 *. stopping_power;
      speed <- speed -|. (axis *|. Vector.dup (Vector.dot speed axis));
      self#change;

  method decelerate (amount : float) = 
    let abs = Vector.abs2 speed in
      if abs > amount then
	begin
	  speed <- speed -|. Vector.unit speed *|. Vector.dup amount;
	  self#change;
	end
      else
	begin
	  if speed <> (0.0, 0.0) then
	    self#change;
	  speed <- (0.0, 0.0)
	end

  method is_alive = hp > 0

  method tick delta world =
    let ratio = 0.01 in
    speed <- 
      Vector.v2 (ratio ** delta) *|. speed +|. 
	  Vector.v2 (1.0 -. ratio ** delta) *|. (cos direction, sin direction) *|. Vector.v2 motor_speed;
(*      if Vector.abs2 speed > 0.0 then
	Printf.printf "%f %f\n%!" (Vector.abs2 speed) motor_speed;*)
    motor_speed <- 0.3 ** delta *. motor_speed;
    at <- at +|. Vector.v2 delta *|. speed;
    firing_time <- firing_time +. delta;
    damage_counter <- max 0.0 (damage_counter *. (0.5 ** delta)-. delta);
    mine_time <- mine_time +. delta;
    let map_friction = 
      List.fold_left 
	(fun friction o -> max friction o#friction)
	0.0
	(world#env (self :> Collision.t) (fun _ -> true)) 
    in
      speed <- 
	(Vector.dup ((max 0.0 (1.0 -. friction -. map_friction)) ** delta) *|. 
	 let abs = Vector.abs2 speed -. opposition *. delta in
	   if abs <= opposition *. delta then
	     0.0, 0.0
	   else
	     (abs, abs) *|. Vector.unit speed);
      if Vector.abs2 speed > 0.0 then
	self#change;
      if self#self_owned then
	begin
	  match world#collision (self :> Collision.t) (fun o -> o#collidable) with
	    | Some (x, objs) ->
		let objs = List.filter (fun o -> o#collide world) objs in
		  if objs <> [] then
		    begin
		      collision_sound world;
		      self#adjust_location x; 
		      self#stop_axis x
		    end
	    | None -> ()
	end

  method get_ammo = ammo

  method get_mines = mines

  method recharge =
    let ops = ref [] in
    let up c h = 
      let op () =
	if c () then begin h (); true; end else false 
      in
	ops := op :: !ops;
    in
      up 
	(fun () -> ammo < max_ammo)
	(fun () -> ammo <- ammo + 1);
      up
	(fun () -> hp < max_hp)
	(fun () -> hp <- hp + 1);
      up
	(fun () -> mines < max_mines)
	(fun () -> mines <- mines + 1);
      List.fold_left (fun recharged op -> op () || recharged) false !ops

  method signal_firing =
    if firing_time > firing_period && ammo > 0 then
      begin
	ammo <- ammo - 1;
	firing_time <- 0.0;
	self#change;
	true;
      end
    else
      false

  method signal_mine = 
    if mine_time > mine_period && mines > 0 then
      begin
	mine_time <- 0.0;
	mines <- mines - 1;
	self#change;
	true;
      end
    else
      false

  method collision_area = 
    (at +|. bitmap_offset, at +|. bitmap_offset +|. bitmap_size), snd (self#cur_frame)

  method cause_damage damage =
    if self#is_alive then
      begin
	hp <- max 0 (hp - damage);
	damage_counter <- damage_counter +. (sqrt (float damage) /. 5.0);
	self#change;
      end

  initializer
    (match init0 with 
       | Object.Init _ -> ()
       | Object.State (state0, age) -> self#set_state state0 age);
    self#change;
end
