(*pp camlp4o -I +sexplib pa_sexp_conv.cmo *)
SEXP_CONV_PATH "Goba"

open VecOps
open Protocol

let port = 61753

let object_increment = 100L

type serial_number = Serial of Int64.t

module PlayerMap = Map.Make(struct type t = Player.id let compare = compare end)

type map_change = Object.id * int

type client = { client_player	: Player.id option;
		client_fd	: Unix.file_descr;
		buffer		: Buffer.t;
		aware		: serial_number;
		latency		: float;
		error		: bool;
		name		: string;
	      }

type player = { player_id	: Player.id;
	      	start_location	: float * float;
		base_id		: Object.object_id;
		tank_ids	: Object.object_id list;
		object_serial	: Int64.t ref;
		object_increment: Int64.t;
	      }

type object' = { object_id	: Object.object_id;
		 obj		: Object.base;
		 parent		: Player.id;
		 changed	: serial_number;
		 parent_aware	: bool;
		 object_type	: ObjectType.t;
		 alive		: bool;
		 transient	: bool;
	       }

type object_event = Create of float * float | Destroy

(*type object_change = { oc_serial	: serial_number;
		       oc_object_id	: Object.id;
		       oc_constr	: object_construction;
		       oc_event		: object_event
		     }
*)

module ClientMap = Map.Make(struct type t = Unix.file_descr let compare = compare end)
module ObjectMap = Map.Make(struct type t = Object.object_id let compare = compare end)

type server = { accept_fd		: Unix.file_descr;
		dgram_fd		: Unix.file_descr;
		mutable clients		: client ClientMap.t;
		mutable players		: player PlayerMap.t;
		mutable ids_avail	: Player.id list;
		mutable objects		: object' ObjectMap.t;
		mutable serial		: serial_number;
		map			: MapData.t;
		map_info		: Tile.tile_info array;
		start_points		: (float * float) list;
		map_changes		: (serial_number * Player.id * map_change) Queue.t }

let player_colors = 
  [(255, 255, 255);
   (255, 0, 0);
   (0, 255, 0);
   (0, 0, 255)]

let available_players server =
  let players = PlayerMap.fold (fun id _ players -> id::players) server.players [] in
    ClientMap.fold
      (fun _ client players ->
	 List.filter (fun player -> Some player <> client.client_player) players
      )
      server.clients
      players

let client_fd client = client.client_fd

let no_serial = Serial 0L

let next_serial =
  let s = ref 1L in
    fun () ->
      s := Int64.add !s 1L;
      Serial !s

let new_object_id =
  let id = ref 1L in
    fun ?player () ->
      let id =
	match player with
	  | None -> id
	  | Some player -> player.object_serial
      in
      let oid = Object.object_id_of_int64 !id in
	id := Int64.add !id object_increment;
	oid	

let create_object otype (obj:Object.base) parent transient = 
  { object_id	= Object.to_object_id obj#get_id;
    obj		= obj;
    parent	= parent;
    changed	= next_serial ();
    parent_aware = false;
    alive	= true; 
    transient	= transient; 
    object_type = otype;
  }

let new_player server player_id =
  let Player.Id player_num = player_id in
  let base_ofs = List.nth server.start_points player_num in
  let color = List.nth player_colors player_num in
  let base = 
    create_object 
      ObjectType.Base
      (new Base.t (new_object_id ()) 
	 (Object.Init { Base.init_at = base_ofs;
			init_color = color }) :> Object.base) 
      player_id false in
  let tanks = 
    let tank ofs =
      create_object 
	ObjectType.Tank
	(new Tank.t (new_object_id ()) 
	   (Object.Init { Tank.init_at = base_ofs +|. ofs;
			  init_color = color;
			  init_direction = 0.0; }) :> Object.base) player_id false
    in
      [
	tank (-.50.0, -.50.0);
	tank (50.0, -.50.0);
	tank (-.50.0, 50.0);
	tank (50.0, 50.0);
      ]
  in
  let objs = 
    base::tanks 
  in
    { player_id		= player_id;
      start_location	= base_ofs; 
      base_id		= base.object_id;
      tank_ids		= List.map (fun t -> t.object_id) tanks;
      object_serial	= ref (Int64.of_int (player_num + 2));
      object_increment	= object_increment;
    },
  objs

let accept_client sock =
  let (fd, _) = Unix.accept sock in
    Unix.set_nonblock fd;
    { client_fd		= fd;
      buffer		= Buffer.create 1024;
      client_player	= None;
      aware		= no_serial;
      latency		= 0.0; 
      error		= false;
      name		= "" }

let create_player t id =
  let player, objects = new_player t id in
    t.players <- PlayerMap.add id player t.players;
    t.objects <- (List.fold_left 
		    (fun map el -> 
		       ObjectMap.add el.object_id el map
		    ) 
		    t.objects 
		    (objects))

let write_eof sock str ofs len =
  match try `Value (Unix.write sock str ofs len) with exn -> `Exn exn with
    | `Value 0 -> raise ProtocolError
    | `Value n -> ()
    | `Exn (Unix.Unix_error ((Unix.ECONNRESET | Unix.EPIPE), _, _)) -> raise ProtocolError
    | `Exn e -> raise e

let send_msg sock msg =
  let str = Sexplib.Sexp.to_string (Protocol.sexp_of_message msg) in
    Unix.clear_nonblock sock;
    write_eof sock str 0 (String.length str);
    write_eof sock "\n" 0 1

let send_client_msg client msg =
  send_msg client.client_fd msg

let handle_map tag server client =
  let msg = tag, Protocol.MsgMap (server.map, server.map_info) in
    send_client_msg client msg;
    client

let handle_ping tag server client =
  send_client_msg client (tag, Protocol.MsgPong);
  client

let handle_objects tag server client =
  client

let handle_list_players tag server client =
  let players =
    PlayerMap.fold (fun id _ players -> id::players) server.players []
  in
    send_client_msg client (tag, Protocol.MsgPlayers players);
    client

let handle_join join tag server client =
  let (name, id) = join in
    if ClientMap.fold (fun _ client ok -> ok && client.client_player <> Some id) server.clients true then
      let player = PlayerMap.find id server.players in
	send_client_msg client (tag, 
				Protocol.MsgPlayerJoined
				  (id, 
				   player.base_id ::
				     List.filter (fun id -> ObjectMap.mem id server.objects) player.tank_ids, 
				   (!(player.object_serial),
				    player.object_increment)));
	let available = List.filter ((<>) id) (available_players server) in
	  ClientMap.iter
	    (fun _ client ->
	       send_client_msg client (Async, MsgAvailablePlayers available)
	    )
	    server.clients;
	  { client with client_player = Some id; name = name }
    else
      begin
	send_client_msg client (tag, Protocol.MsgPlayerJoinFailed);
	client
      end

let find_object server object_id =
  try Some (ObjectMap.find object_id server.objects)
  with Not_found -> None

let handle_object_info (object_id, object_state) tag server client =
  let object_id = Object.to_object_id object_id in
    begin
      match find_object server object_id with
	| Some o ->
	    (*    Printf.printf "%d %d\n%!" 
		  (match client.client_player with Some (Player.Id i) -> i | _ -> -1)
		  (match o.parent with Player.Id i -> i);*)
	    o.obj#set_state object_state (client.latency /. 2.0);
	    server.objects <-
	      ObjectMap.add object_id 
	      { o with 
		  changed = next_serial ();
		  parent_aware = client.client_player = Some o.parent;
	      }
	      (ObjectMap.remove object_id server.objects)
	| None -> ()
    end;
    client

let handle_create_object { oi_id = object_id;
			   oi_pid = player_id;
			   oi_state = object_state;
			   oi_type = object_type } tag server client =
  let id = Object.to_object_id object_id in
  let delay = 0.0 in
  let obj =
    match object_type with
      | ObjectType.Base ->
	  let obj = new Base.t id (Object.State (object_state, delay)) in
	    (obj :> Object.base);
      | ObjectType.Tank ->
	  let obj = new Tank.t id (Object.State (object_state, delay)) in
	    (obj :> Object.base);
      | ObjectType.Projectile -> 
	  let obj = new Projectile.t id (Object.State (object_state, delay)) in
	    (obj :> Object.base)
      | ObjectType.Explosion ->
	  let obj = new Explosion.t id (Object.State (object_state, delay)) in
	    (obj :> Object.base)
      | ObjectType.Mine ->
	  let obj = new Mine.t id (Object.State (object_state, delay)) in
	    (obj :> Object.base)
  in
    server.objects <-
      ObjectMap.add id
      { object_id = id;
	obj = obj;
	object_type = object_type;
	changed = next_serial ();
	parent = player_id;
	(*check against? (match client.client_player with Some x -> x | None -> failwith "Added object without joining")*)
	parent_aware = true;
	alive = true;

	(* kludge *)
	transient = (match object_type with ObjectType.Mine -> false | _ -> true);
      }
      (ObjectMap.remove id server.objects);
    client

let destruct_object server origin object_id =
  match find_object server object_id with
    | Some o -> 
	server.objects <-
	  ObjectMap.add object_id 
	  { o with 
	      changed = next_serial ();
	      alive = false;
	      parent_aware = 
	      (match origin with 
		 | Some { client_player = Some player } when player = o.parent -> true
		 | _ -> false);
	  }
	  (ObjectMap.remove object_id server.objects)
    | None -> ()

let object_list server f =
  List.rev (
    ObjectMap.fold
      (fun _ obj list ->
	 if f obj then
	   obj::list
	 else
	   list
      )
      server.objects
      []
  )

let object_id_list server f =
  List.map (fun { object_id = id } -> id) (object_list server f)

let destruct_transient_objects server client =
  List.iter (destruct_object server None) 
    (object_id_list server 
       (function 
	  | { transient = true; parent = parent } when Some parent = client.client_player -> true
	  | _ -> false))

let handle_destruct_object object_id tag server client =
  let object_id = Object.to_object_id object_id in
    destruct_object server (Some client) object_id;
    client

let handle_client_latency latency tag server client =
  { client with latency = latency }

let handle_map_change ((x, y), damage) tag server client =
  if x < 0 || y < 0 || x >= fst (MapData.dims server.map) || y >= snd (MapData.dims server.map) then
    client
  else
    begin
      Queue.add (next_serial (), (match client.client_player with Some id -> id | _ -> failwith "argh"), (Object.MapId (x, y), damage)) server.map_changes;
      let (w, h) = MapData.dims server.map in
	server.map_info.(w * y + x) <- { Tile.hp = server.map_info.(w * y + x).Tile.hp - damage };
	client
    end

let handle_damage_object (obj_damage : Object.id * int) tag server client =
  match obj_damage with
    | Object.ObjectId id, damage ->
	ClientMap.iter
	  (fun _ client' ->
	     if client'.client_player <> client.client_player then
	       send_client_msg client' (Async, MsgDamageObject obj_damage)
	  )
	  server.clients;
	client
    | Object.MapId at, damage ->
	handle_map_change (at, damage) tag server client

let handle_chat chat tag server client =
  ClientMap.iter
    (fun _ client' ->
       send_client_msg client' (tag, MsgChat (client.name, chat))
    )
    server.clients;
  client

let handle_query_available_players tag server client =
  send_client_msg client (tag, MsgAvailablePlayers (available_players server));
  client

let handle_command server client cmd =
  let cmd = 
    if String.length cmd > 0 && cmd.[String.length cmd - 1] = '\r' then
      String.sub cmd 0 (String.length cmd - 1)
    else
      cmd 
  in
  let tag, f = 
    let tag, msg =
      let handle_error msg =
	Printf.printf "Sexp error: %s\n%!" msg;
	raise ProtocolError
      in
	try Protocol.command_of_sexp (Sexplib.Sexp.of_string cmd)
	with 
	    (* kludge for bad error reporting from Sexplib *)
	  | Failure msg when let error = "Sexp.of_string:" in String.sub msg 0 (String.length error) = error ->
	      handle_error msg
	  | Sexplib.Conv.Of_sexp_error (msg, _) ->
	      handle_error msg
    in
      tag,
      match msg with
	| Protocol.CmdMap -> handle_map
	| Protocol.CmdObjects -> handle_objects
	| Protocol.CmdPing -> handle_ping
	| Protocol.CmdListPlayers -> handle_list_players
	| Protocol.CmdJoin join -> handle_join join
	| Protocol.CmdObjectInfo info -> handle_object_info info
	| Protocol.CmdReportLatency latency -> handle_client_latency latency
	| Protocol.CmdCreateObject info -> handle_create_object info
	| Protocol.CmdDestructObject oid -> handle_destruct_object oid
	| Protocol.CmdDamageObject obj_damage -> handle_damage_object obj_damage
	| Protocol.CmdChat chat -> handle_chat chat
	| Protocol.CmdQueryAvailablePlayers -> handle_query_available_players
  in
    f tag server client


let rec handle_buffer server client msg =
  GobaUtils.fix_buffer client.buffer msg 
    (fun (response, client) msg ->
       match try `Continue, handle_command server client msg with ProtocolError -> `Disconnect, client with
	 | `Disconnect, client -> `Stop, (`Disconnect, client)
	 | `Continue, client -> `Continue, (`Continue, client)
    )
    (`Continue, client)

let handle_client server client =
  let buf = String.make 1024 ' ' in
    Unix.set_nonblock client.client_fd;
    match GobaUtils.valuefy (Unix.read client.client_fd buf 0) 1024 with
      | `Exn (Unix.Unix_error ((Unix.EINTR | Unix.EWOULDBLOCK), _, _)) ->
	  `Disconnect, client
      | `Exn (Unix.Unix_error (_, _, _))
      | `Value 0 ->
	  `Disconnect, client
      | `Exn exn -> raise exn
      | `Value got ->
	  (* inefficient, but easy *) 
	  handle_buffer server client (String.sub buf 0 got)
	

let client_finish client =
  Unix.close client.client_fd

let send_object_info sock o =
  let msg = 
    if o.alive then
      Protocol.MsgReportObject (
	{ oi_id = Object.ObjectId o.object_id;
	  oi_pid = o.parent;
	  oi_state = o.obj#get_state 0.0;
	  oi_type = o.object_type }
      )
    else
      Protocol.MsgDestructObject (Object.ObjectId o.object_id)
  in
    send_msg sock (Protocol.Async, msg)

let lowest_serial t =
  ClientMap.fold
    (fun _ { aware = aware } serial ->
       min aware serial
    )
    t.clients
    (next_serial ())

let handle_client_objects t client =
  if client.client_player <> None then
    begin
      t.objects <-
	ObjectMap.fold
	(fun id o objs ->
	   if o.changed > client.aware && 
	     (not o.parent_aware || client.client_player <> Some o.parent) then
		 begin
		   (*if client.client_player = Some (Player.Id 0) then
		   Printf.printf "Client %s aware %Ld, object %s parented by %s changed %Ld\n%!"
		   (match client.client_player with Some i -> Player.string_of_id i | None -> assert false) 
		   (match client.aware with Serial s -> s) 
		   (Object.string_of_id (Object.ObjectId o.object_id))
		   (Player.string_of_id o.parent)
		   (match o.changed with Serial s -> s);*)
		   send_object_info client.client_fd o;
		 end;
	   ObjectMap.add id 
	     { o with parent_aware = 
		 if not o.parent_aware then false
		 else if client.client_player = Some o.parent then false
		 else true } objs
	)
	t.objects
	t.objects;
    end;
  client

let handle_client_fds t rd =
  t.clients <-
    List.fold_left
    (fun clients fd ->
       match
	 try Some (ClientMap.find fd clients)
	 with Not_found -> None
       with
	 | None -> clients
	 | Some client -> 
	     let clients = 
	       ClientMap.remove (client_fd client) clients
	     in
	       match handle_client t client with
		 | `Disconnect, client -> 
		     destruct_transient_objects t client; 
		     client_finish client;
		     clients
		 | _, client' -> 
		     ClientMap.add 
		       (client_fd client) 
		       client'
		       clients
    ) t.clients rd

let handle_map t =
  Queue.iter 
    (fun (_, player, change) -> 
       ClientMap.iter 
	 (fun _ client ->
	    if client.client_player <> Some player then
	      send_client_msg client (Protocol.Async, Protocol.MsgDamageObject change);
	 )
	 t.clients
    )
    t.map_changes;
  Queue.clear t.map_changes

let handle_objects t =
  t.clients <-
    ClientMap.map
    (fun client -> 
       try handle_client_objects t client
       with ProtocolError -> { client with error = true })
    t.clients;
  t.objects <-
    ObjectMap.fold
      (fun oid obj new_objects ->
	 if not obj.alive then
	   ObjectMap.remove oid new_objects
	 else
	   new_objects)
      t.objects
      t.objects

let update_serials t =
  let serial = next_serial () in
    t.clients <- ClientMap.map (fun client -> 
				  if client.client_player <> None then
				    { client with aware = serial }
				  else
				    client ) t.clients

let handle_dead_clients t =
  t.clients <-
    ClientMap.fold
    (fun key client clients ->
       match client.error with
	 | false -> clients
	 | true -> 
	     destruct_transient_objects t client; 
	     client_finish client;
	     ClientMap.remove key clients
    )
    t.clients
    t.clients

let server_loop t =
  let rec loop () = 
    let (rd, _, _) = Unix.select ([t.accept_fd] @ ClientMap.fold (fun fd _ fds -> fd::fds) t.clients []) [] [] ~-.1.0 in
      if List.mem t.accept_fd rd then
	let c = accept_client t.accept_fd in
	  t.clients <- ClientMap.add (client_fd c) c t.clients
      else
	();
      handle_client_fds t rd;
      handle_objects t;
      handle_map t;
      update_serials t;
      handle_dead_clients t;
      loop ()
  in
    loop ()
    
let main () =
  let accept_fd = Net.create_server_socket port in
  let dgram_fd = Net.create_udp_socket port in
(*  let map = MapFile.load_map "map.txt" in*)
  let map = MapGenerator.generate (80, 80) (20, 20) in
  let start_points = 
    List.map snd (
      List.sort compare (
	GobaUtils.foldl2d
	  (fun at points ->
	     match MapData.get map at with
	       | Tile.Start x ->
		   MapData.set map at Tile.Space;
		   (x, (Vector.float2 at *|. Vector.float2 (MapData.tile_size map)))::points
	       | _ -> points)
	  []
	  ((0, 0), MapData.dims map)
      )
    ) in
  let ids_avail = List.rev (GobaUtils.foldl1d (fun v l -> Player.Id v::l) [] (0, List.length start_points)) in
  let t = { accept_fd = accept_fd;
	    dgram_fd = dgram_fd;
	    clients = ClientMap.empty;
	    players = PlayerMap.empty;
	    ids_avail = ids_avail;
	    objects = ObjectMap.empty;
	    serial = next_serial ();
	    map = map;
	    start_points = start_points;
	    map_info = Array.map (fun tile -> snd (Tile.tile_with_info tile)) map.MapData.data;
	    map_changes = Queue.create () }
  in
    GobaUtils.foldl1d (fun n () -> create_player t (Player.Id n)) () (0, 4);
    server_loop t

let _ = 
  let _ = Sdl.init [`EVERYTHING] in
  if not !Sys.interactive then
    let _ = Sys.signal Sys.sigpipe Sys.Signal_ignore in
      Random.self_init ();
      main ()

