SEXP_CONV_PATH "Goba"

exception ProtocolError

type serial = Serial of int64
with sexp

type object_location = float * float 
with sexp

type object_speed = float * float 
with sexp

type tag = Tag of int64
with sexp

let no_tag = Tag 0L

let new_tag =
  let t = ref 0L in
    fun () ->
      let tag = Tag !t in
	t := Int64.add !t 1L;
	tag

type object_info = 
    { oi_id : Object.id;
      oi_pid : Player.id;
      oi_type : ObjectType.t;
      oi_state : Sexplib.Sexp.t;
    }
with sexp

type chat_message = string
with sexp

type chat = string * chat_message
with sexp

type available_players = Player.id list
with sexp

  (* player id, base id, tank ids, object id base and increment *)
type join_info = (Player.id * Object.object_id list * (int64 * int64))
with sexp

type message' =
  | MsgReportObject of object_info
  | MsgDestructObject of Object.id
  | MsgMap of (MapData.t * Tile.tile_info array)
  | MsgPong
  | MsgPlayers of (Player.id) list
  | MsgSuccess
  | MsgDamageObject of (Object.id * int)
  | MsgChat of chat
  | MsgAvailablePlayers of available_players

  | MsgPlayerJoined of join_info
  | MsgPlayerJoinFailed
with sexp

type cmd_join = string * Player.id
with sexp

type command' =
  | CmdMap
  | CmdObjects
  | CmdPing
  | CmdListPlayers
  | CmdJoin of cmd_join
  | CmdCreateObject of object_info
  | CmdObjectInfo of (Object.id * Sexplib.Sexp.t)
  | CmdDestructObject of Object.id
  | CmdReportLatency of float
  | CmdDamageObject of (Object.id * int)
  | CmdChat of chat_message
  | CmdQueryAvailablePlayers
with sexp

type sync = 
  | Sync of tag
  | Async
with sexp

type command = sync * command'
with sexp

type 'msg datagram =
  | Datagram of (serial * 'msg)
with sexp

type message = (sync * message')
with sexp

let string_of_message m =
  Sexplib.Sexp.to_string (sexp_of_message m)

let string_of_command m =
  Sexplib.Sexp.to_string (sexp_of_command m)

let read_response buffer of_sexp sock =
  let buf = String.create 1024 in

  let receive () =
    match GobaUtils.valuefy (Unix.read sock buf 0) (String.length buf) with
      | `Value 0 -> raise ProtocolError
      | `Value n -> Some (String.sub buf 0 n)
      | `Exn (Unix.Unix_error (Unix.EAGAIN, _, _)) -> None
      | `Exn e -> raise e
  in
  let rec wait_sync rs =
    match receive () with
      | None -> List.rev rs
      | Some str ->
	  match
	    GobaUtils.fix_buffer buffer str
	      (fun (got_sync, rs) line ->
		 let msg = of_sexp (Sexplib.Sexp.of_string line) in
		 let got_sync = got_sync || match msg with Sync _, _ -> true | Async, _ -> false in
		   `Continue, (got_sync, msg::rs)
	      )
	      (false, rs)
	  with
	    | true, rs -> List.rev rs
	    | false, rs -> wait_sync rs
  in
    wait_sync []

