SEXP_CONV_PATH "Goba"

open VecOps

let bitmap = lazy (Bitmaps.bitmap (`Block (5, 5, (255, 0, 0), 255)))
let cmask = lazy (Bitmaps.collision_mask_of_surface (snd (Lazy.force bitmap)))

let hit_sound = Sfx.Sample "hit"

let orientation_speed = 10.0

let channel = Sfx.alloc_channel ()

type init = { init_orientation  : float * float;
	      init_parent	: Object.id;
	      init_at		: float * float;
	      init_speed	: float * float; }
with sexp

class t id init0 =
  let init = Object.default_with init0 init_of_sexp in
  let orientation = init.init_orientation in
  let parent = init.init_parent in
  let normal = Vector.coaxel (Vector.unit orientation) in

object (self)
  inherit Object.t id as super

  val mutable at = init.init_at
  val mutable speed = init.init_speed
  val mutable alive = true

  val mutable distance = 90.0

  method location = at
  method z_location = 0.0

  method speed = speed
  method set_speed speed' = speed <- speed'

  method set_location at' = at <- at'

  method is_alive = alive

  method get_state delay =
    Object.sexp_of_state sexp_of_init
      { Object.state_super = super#get_state delay;
	state_init = init }

  method set_state state delay =
    let state = Object.state_of_sexp init_of_sexp state in
      super#set_state state.Object.state_super delay
 
 method render (gfx:#Gfx.t) _ at =
    if alive then
      let ofs, surface = Lazy.force bitmap in
	gfx#blit (at +| ofs) surface

  method detonate explosion =
    begin
      match explosion with
	| Some ((world : Object.world), direction) -> 
	    world#explode at direction;
	    (world#channel channel)#play hit_sound
	| None -> ();
    end;
    alive <- false;
    self#change;

  method tick delta (world:Object.world) =
    if alive then
      begin
	distance <- 
	  distance -. 
	      Vector.abs2 (speed /|. (1.0, (* tank viewport ratio *) 180.0 /. 195.0)) *. delta;
	if distance < 0.0 then
	  self#detonate (Some (world, (0.0, 0.0)))
	else
	  begin
	    self#change;
	    at <- at +|. speed *|. (delta, delta);
	    speed <- speed -|. 
		normal *|. Vector.dup (Vector.dot speed normal *. delta *. orientation_speed);
	    match world#collision (self :> Collision.t) (fun o -> o#get_id <> parent && o#destructible) with
	      | None -> ()
	      | Some (x, objs) -> 
		  if self#self_owned then
		    let rec give_damages damage_left damages =
		      let damages, _, damage_left =
			List.fold_left
			  (fun (left, right, damage_left) obj ->
			     let r = List.hd right in
			       if damage_left > 0 then
				 (r + 1::left, List.tl right, damage_left - 1)
			       else
				 (r::left, List.tl right, damage_left)
			  )
			  ([], damages, damage_left)
			  objs
		      in
		      let damages = List.rev damages in
			if damage_left > 0 then
			  give_damages damage_left damages
			else
			  damages
		    in
		    let damages = give_damages 5 (List.map (fun _ -> 0) objs) in
		    let l = Vector.abs2 x in
		    let u = (Vector.unit x -|. Vector.unit speed) /|. Vector.dup 2.0 in
		      List.iter2 (fun obj damage -> obj#impact world damage) objs damages; 
		      self#detonate (Some (world, u *|. Vector.dup l))
		  else
		    self#detonate (Some (world, (0.0, 0.0)))
	  end
      end

  method collision_area =
    (at, at +|. (5.0, 5.0)), Lazy.force cmask

  method cause_damage _ =
    self#detonate None

  method collidable = false

  initializer
    (match init0 with
       | Object.Init _ -> ()
       | Object.State (state0, age) -> self#set_state state0 age);
    self#change;
end
