SEXP_CONV_PATH "Goba"

open VecOps

type object_id = int64
with sexp

type map_id = (int * int)
with sexp

type id = 
  | MapId of map_id
  | ObjectId of object_id
with sexp

exception NotValidObjectType
let to_object_id = function
  | ObjectId i -> i
  | _ -> raise NotValidObjectType

let to_map_id = function
  | MapId i -> i
  | _ -> raise NotValidObjectType

let object_id_of_int64 id = id

let string_of_id = function
  | MapId (x, y) -> Printf.sprintf "MapId (%d,%d)" x y
  | ObjectId id -> Printf.sprintf "ObjectId %Ld" id

type serial = Serial of int64

let next_serial =
  let serial = ref 0L in
    fun () ->
      let s = Serial !serial in
	serial := Int64.add !serial 1L;
	s

let no_serial = Serial 0L

type render_flag = [ `Labels | `LabelsExcept of id list ]
type render_flags = render_flag list

class type base =
object
  method get_state : float -> Sexplib.Sexp.t
  method set_state : Sexplib.Sexp.t -> float -> unit
  method collision_area : ((float * float) * (float * float)) * CollisionMask.mask
  method self_owned : bool
  method get_id : id
  method location : float * float
  method z_location : float
  method speed : float * float
  method render : Gfx.t -> render_flags -> int * int -> unit
  method impact : world -> int -> unit
  method collidable : bool
  method destructible : bool
  method set_self_owned : bool -> unit
  method is_alive : bool

  (* used by networking *)
  method cause_damage : int -> unit

  method friction : float

  method changed_since : serial -> bool
  method change : unit

  method tick : float -> world -> unit

  method recharge : bool

  method collide : world -> bool
end and
  world =
object
  method collision : Collision.t -> (base -> bool) -> ((float * float) * base list) option
  method explode : float * float -> float * float -> unit
  method env : Collision.t -> (base -> bool) -> base list
  method objs : (float * float) * (float * float) -> (base -> bool) -> base list
  method report_damage : id * int -> unit
  method channel : Sfx.channel_number -> Sfx.channel
end

type place_motion = ((float * float) * (float * float))
with sexp

type 'a init =
  | State of (Sexplib.Sexp.t * float)
  | Init of 'a

type 'a state = { state_super : Sexplib.Sexp.t;
		  state_init  : 'a }
with sexp

let default init v =
  match init with
    | State _ -> v
    | Init v' -> v'

let default_by init f =
  match init with
    | State s -> f s
    | Init v' -> v'

let default_with init f =
  match init with
    | State (state, _) -> (state_of_sexp f state).state_init
    | Init v' -> v'

class virtual t (id : object_id) = 
object (self : 'a)
  val mutable self_owned = false
  val mutable change_serial = no_serial

  method get_state delay =
    sexp_of_place_motion (self#location +|. self#speed *|. Vector.dup delay, 
			  self#speed)
  method set_state state age =
    let (at, speed) = place_motion_of_sexp state in
      self#set_location (at +|. speed *|. Vector.dup age);
      self#set_speed speed

  method virtual collision_area : ((float * float) * (float * float)) * CollisionMask.mask
  method virtual is_alive : bool
  method virtual location : float * float
  method z_location = 0.0
  method virtual render : Gfx.t -> render_flags -> int * int -> unit
  method private virtual set_location : float * float -> unit
  method private virtual set_speed : float * float -> unit
  method virtual speed : float * float

  method set_self_owned s = self_owned <- s
  method self_owned = self_owned
  method get_id = ObjectId id
  method collidable = true
  method destructible = self#collidable

  method cause_damage (_:int) = ()

  method impact (world:world) (damage:int) = 
    world#report_damage (self#get_id, damage);
    self#cause_damage damage
  method collide (_:world) = self#collidable

  method change = change_serial <- next_serial ()

  method friction = 0.0

(* if you've never called change, assume the object doesn't support
   change notifications and hence will be considered always changed *)

  method tick (_:float) (_:world) = ()

  method recharge = false

  method changed_since s = 
    change_serial = no_serial || 
    change_serial > s
end
