SEXP_CONV_PATH "Goba"

open VecOps

let damage = 40

let radius = 40

let bitmap_nonactive = 
  lazy (let (ofs, bmap) = Bitmaps.bitmap (`Block (5, 5, (255, 0, 0), 255)) in
	  (ofs -| (2, 2), bmap))

(*let bitmap_active = 
  lazy (let (ofs, bmap) = Bitmaps.bitmap (`Block (5, 5, (128, 128, 128), 255)) in
	  (ofs -| (2, 2), bmap))*)

let cmask = lazy (fst (Lazy.force bitmap_nonactive), Bitmaps.collision_mask_of_surface (snd (Lazy.force bitmap_nonactive)))

(*let cofs, cmask = CollisionMask.circle_loc_mask radius*)

let explosion_sound = Sfx.Sample "explosion"
let activation_sound = Sfx.Sample "mine-activated"
let triggered_sound = Sfx.Sample "mine-triggered"

let flash_time = 0.2

let channel = Sfx.alloc_channel ()

type init = { init_at : float * float }
with sexp

class t id init0 =
  let init = Object.default_with init0 init_of_sexp in
  let flash_cycle = sqrt (Random.float 20.0) +. 5.0 in
object (self)
  inherit Object.t id as super
    
  val mutable at = init.init_at
  val mutable alive = true
  val mutable trig_time = 1.0 
  val mutable activated = false
  val mutable explosion_time = None
  val mutable flash_counter = Random.float flash_cycle

  method location = at
  method speed = (0.0, 0.0)
  method z_location = 0.0

  method get_state delay =
    Object.sexp_of_state sexp_of_init
      { Object.state_super = super#get_state delay;
	state_init = init }

  method set_state state delay =
    let state = Object.state_of_sexp init_of_sexp state in
      super#set_state state.Object.state_super delay

  method set_speed _ = ()
  method set_location at' = at <- at'

  method is_alive = alive

  method render gfx _ at =
    if (self#self_owned && not activated) || 
      explosion_time <> None ||
      flash_counter < flash_time then
      let bitmap = bitmap_nonactive in
      let ofs, surface = Lazy.force bitmap in
	gfx#blit (at +| ofs) surface

  method tick delta world =
    if alive then
      begin
	flash_counter <- mod_float (flash_counter +. delta) flash_cycle;
	if not activated then
	  begin
	    trig_time <- trig_time -. delta;
	    activated <- trig_time <= 0.0;
	    if activated then
	      (world#channel channel)#play activation_sound
	  end
	else
	  begin
	    match explosion_time with
	      | None -> ()
	      | Some t when t < 0.0 ->
		  self#detonate world
	      | Some t -> 
		  explosion_time <- Some (t -. delta)
	  end
      end

  method destructible = false

  method collision_area =
    let offset = Vector.float2 (fst (Lazy.force cmask)) in
      ((at +|. offset, 
	at +|. offset +|. Vector.float2 (CollisionMask.size (snd (Lazy.force cmask)))), 
       snd (Lazy.force cmask))
	
  method trigger (world:Object.world) =
    (world#channel channel)#play triggered_sound;
    explosion_time <- Some 0.1

  method detonate world =
    (world#channel channel)#play explosion_sound;
    self#change;
    self#set_self_owned true;
    alive <- false;
    let objs = 
      world#objs 
	(at -|. Vector.dup (float radius *. 2.0),
	 at +|. Vector.dup (float radius *. 2.0))
	(fun obj -> Vector.abs2 (obj#location -|. at) <= float radius)
    in
      List.iter (fun o -> 
		   let distance = (Vector.abs2 (o#location -|. at) /. float radius) +. 1.0 in
		     o#impact world (int_of_float (float damage /. distance))) objs;
      world#explode at (0.0, 0.0)

  method cause_damage damage =
    alive <- false

  method impact (world : Object.world) _ =
    if alive then
      self#trigger world

  method collide world =
    if alive && activated && explosion_time = None then
      begin
	self#trigger world;
	true
      end
    else
      false

  initializer
    (match init0 with 
       | Object.Init _ -> ()
       | Object.State (state0, age) -> self#set_state state0 age);
    self#change;
end
