open VecOps
open Tile

module N = PerlinNoise

type kind = Tile.tile

let flip a b c = a c b

(* used by currying, thus the reversed args *)
let gt a b = b > a
let lt a b = b < a

let const a b = a

let ranges =
  [
    Mountain,	lt 100;
    Grass,	lt 120;
    Water,	lt 100;

    Space,	const true;
  ]

let segment len step =
  let rec do_segment at left =
    if left > 0 then
      [< 'at; do_segment (at +| step) (left - 1) >]
    else
      [< >]
  in
    do_segment (0, 0) len

let rec offset ofs st =
  match st with parser
    | [< 'at >] -> [< 'at +| ofs; offset ofs st >]
    | [< >] -> [< >]

let rec list_of_st ?n s = 
  match s with parser
    | [< 'p >] -> p::(list_of_st s)
    | [< >] -> []

let rec take n s =
  if n > 0 then
    match s with parser
      | [< 'p >] -> [< 'p; take (n - 1) s >]
      | [< >] -> [< >]
  else
    [< >]

let head_st st = 
  match st with parser
    | [< 'v >] -> v
    | [< >] -> failwith "Empty stream" 

let rec concat_stf st1 f_st2 =
  match st1 with parser
    | [< 'x >] -> [< 'x; concat_stf st1 f_st2 >]
    | [< >] -> f_st2 ()

let rec concat_st st1 st2 =
  match st1 with parser
    | [< 'x >] -> [< 'x; concat_st st1 st2 >]
    | [< >] -> st2

let spiral () =
  let (@@) = concat_stf in
  let rec sp d0 =
    segment d0 (1, 0) @@
      (fun () -> offset (d0, 0) (segment d0 (0, 1)) @@
	 (fun () -> offset (d0, d0) (segment d0 (-1, 0)) @@
	    (fun () -> offset (0, d0) (segment d0 (0, -1)) @@
	       (fun () -> offset (-1, -1) (sp (d0 + 2))
	       )
	    )
	 )
      )
  in
    sp 1

let rec filter_st f st =
  match st with parser
    | [< 'v >] ->
	if f v then
	  [< 'v; filter_st f st >]
	else
	  filter_st f st
    | [< >] -> [< >]

let rec find_st f st =
  match st with parser
    | [< 'v >] ->
	if f v then
	  v
	else
	  find_st f st
    | [< >] -> raise Not_found

let inside_map (width, height) (x, y) =
  x >= 0 && y >= 0 && x < width && y < height

let generate width height =
  let map = Array.create (width * height) Space in
  let mountains = N.create ~scale:((+) 20) () in
  let water = N.create () in
  let grass = N.create () in
    for x = 0 to width - 1 do
      for y = 0 to height - 1 do
	let at = (x, y) in
	let m = N.at mountains (at *| (2, 2)) in
	let w = N.at water (at *| (2, 2)) in
	let g = N.at grass (at *| (3, 3)) in
	let value =
	  let (tile, _) =
	    List.find (
	      fun (tile, f) -> 
		let v = 
		  match tile with
		    | Mountain -> m
		    | Grass -> g
		    | Water -> w
		    | Space -> 0
		    | _ -> failwith "Hmm"
		in
		  f v
	    ) ranges
	  in
	    tile
	in
	  map.(y * width + x) <- value;
      done
    done;
    fun (x, y) -> map.(y * width + x)

let find_free_areas threshold (width, height, map) =
  let wall = Array.create (width * height) false in
  let set_wall (x, y) = 
    if inside_map (width, height) (x, y) then
      wall.(y * width + x) <- true in
  let wall_at at = not (inside_map (width, height) at) || not (map at) in
  let mark at =
    if wall_at at then
      begin
	for xc = -threshold to threshold do
	  for yc = -threshold to threshold do
	    set_wall (at +| (xc, yc));
	  done
	done
      end
  in
    for x = -1 to width do mark (x, -1); mark (x, height) done;
    for y = 0 to height - 1 do mark (-1, y); mark (width, y) done;

    for y = 0 to height - 1 do
      for x = 0 to width - 1 do mark (x, y) done;
    done;
    fun (x, y) -> not wall.(y * width + x)

let within a low high = a >= low && a < high

let zoneify : (int * int) -> ((int * int) -> 'a) -> ((int * int) list) -> ((int * int) list * ((int * int) -> int option)) =
  fun (width, height) map start_points ->
    let zones = Array.create (width * height) None in
    let visited = Array.create (width * height) false in
    let has_visited (x, y) = visited.(y * width + x) in
    let set_visited (x, y) = visited.(y * width + x) <- true in
    let get_zone (x, y) = zones.(y * width + x) in
    let set_zone (x, y) zone = zones.(y * width + x) <- zone in
    let dirs = [-1, -1; -1, 0; -1, 1;
		0, -1;          0, 1;
		1, -1;   1, 0;  1, 1] in
    let rec fill_at zone at old_color =
      if within (fst at) 0 width &&
	within (snd at) 0 height &&
	not (has_visited at) && (old_color = map at) then
	  begin
	    set_zone at (Some zone);
	    set_visited at;
	    List.fold_left (fun filled ofs -> filled + fill_at zone (at +| ofs) old_color) 1 dirs;
	  end
      else
	0
    in
    let zone_sizes =
      List.rev (
	snd (
	  List.fold_left
	    (fun (zone, zones) at -> 
	       let size = fill_at zone at (map at) in
		 if size > 0 then
		   (zone + 1, (zone, size)::zones)
		 else
		   (zone, zones)
	    )
	    (0, [])
	    start_points
	)
      )
    in
      (zone_sizes, get_zone)

(* yes, this fails if no center is found with division by zero *)
let mass_center (width, height, map) =
  let count, sum =
    GobaUtils.foldl2d 
      (fun at (count, sum) -> if map at then (count + 1, sum +| at) else (count, sum))
      (0, (0, 0))
      ((0, 0), (width, height))
  in
    sum /| Vector.v2 count

let find_nearest (width, height, map) at =
  head_st
    (filter_st 
       (fun at -> 
	  inside_map (width, height) at && map at
       ) 
       (offset at (spiral ())))

let replace f v value =
  fun v' ->
    if v = v' then
      value
    else
      f v'

let shuffle l =
  List.map snd (
    List.sort compare (
      List.map
	(fun el ->
	   Random.float 1.0, el)
	l
    )
  )
let generate size tile_size = 
  let width, height = size in
  let map = generate width height in
  let free_map = 
    find_free_areas 6
      (width, height, 
       (fun at -> 
	  List.exists ((=) (map at)) [Grass; Space])) in
  let zones, zone_map = 
    let start_points =
      (GobaUtils.foldl2d
	 (fun at free ->
	    if free_map at then
	      at::free
	    else
	      free
	 )
	 []
	 ((0, 0), (width, height)))
    in
      zoneify 
	(width, height) 
	(fun at -> map at <> Mountain)
	start_points
  in
  let biggest_zone =
    List.fold_left
      (fun (best_zone, best_size) (zone, size) ->
	 if size > best_size then
	   (zone, size)
	 else
	   (best_zone, best_size))
      (List.hd zones)
      (List.tl zones)
  in
  let free_map = 
    let v = Array.create (width * height) false in
      GobaUtils.iter2d
	(fun ((x, y) as at) ->
	   v.(y * width + x) <- 
	     free_map at && zone_map at = Some (fst biggest_zone)
	)
	((0, 0), (width, height));
      fun (x, y) -> v.(y * width + x)
  in
  let points = 
    (*let center = mass_center (width, height, free_map) in*)
    let corners =
      [(0, 0); (width, 0); (0, height); (width, height)] in
    let points =
      List.map
	(fun corner -> 
	   find_nearest (width, height, free_map) 
	     (*((corner +| center) /| (2, 2))*)
	     corner
	)
	corners
    in
    let numbers = snd (List.fold_left (fun (n, nums) _ -> (n + 1, n::nums)) (0, []) points) in
      List.combine (shuffle numbers) points
  in
  let map =
    List.fold_left
      (fun map (n, point) ->
	 replace map point (Start n))
      map
      points
  in
(*
    if snd (biggest_zone) < width * height / 4 then
      Printf.printf "Warning: biggest zone is very little! (Less than 1/4 of the map area)\n";
    Printf.printf "%d\n%d\n" width height;
    Printf.printf "%d\n%d\n" 11 18;
    dump_map (width, height, (fun at -> char_of_tile (map at)));
    Printf.printf "Free map\n%!";
    dump_map (width, height, (fun at -> if free_map at then '#' else '.'));
    Printf.printf "Zone map\n%!";
    dump_map (width, height, (fun at -> 
				match zone_map at with
				  | None -> ' '
				  | Some x -> Char.chr (Char.code '0' + x)));
    (*    List.iter
	  (fun (x, y) ->
	  Printf.printf "%d, %d\n" x y)
	  points*)
    ()*)
    MapData.create ~f:map (width, height) tile_size
