open VecOps
open GobaUtils

type t = { width	: int;
	   height	: int;
	   contents	: Tile.t array;
	   bitmaps	: ((int * int) * Sdlvideo.surface) option array;
	   tile_size	: int * int;
	 }

let black = (0, 0, 0)
let yellow = (255, 255, 200)
let white = (255, 255, 255)
let grey = (128, 128, 128)
let dark_grey = (64, 64, 64)
let green = (0, 255, 0)
let blue = (128, 128, 255)
let brown = (139, 69, 19)

let clamp low high v = min high (max low v)

let scale low high v =
  float (v - low) /. float (high - low)

let ipolate3 v col1 col2 =
  let v = clamp 0.0 1.0 v in
  let s = Vector.v3 v in
  let s'rev = Vector.v3 (1.0 -. v) in
    Vector.int3 (s'rev *||. Vector.float3 col1 +||. s *||. Vector.float3 col2)

let color_map (tile, info) = 
  let hp = info.Tile.hp in
    match tile with
      | Tile.Space -> white
      | Tile.Grass -> ipolate3 (scale 10 0 hp) green black
      | Tile.Water -> blue
      | Tile.Mountain -> ipolate3 (scale 50 0 hp) grey dark_grey
      | Tile.Wall -> white
      | Tile.Start _ -> failwith "No color for start place"

(* offset is used for multi-block tiles *)
let create_tile offset get =
  let mod_tile base =
    let xmod, ymod = fst offset mod 3, snd offset mod 3 in
      `Move ((0, 20), `Bitmap (Printf.sprintf "%s-%d%d" base xmod ymod))
  in
  let grass_tile () =
    mod_tile "grass"
      (*    `Move ((0, 0), `Bitmap "grass")*)
  in
  let space_tile () = 
    mod_tile "space"
      (*`Empty*)
  in
  let (tile, info) = get offset in
  let masks =
    [(-1, 0), `Bitmap "partial-mask-w";
     (0, -1), `Bitmap "partial-mask-n";
     (1, 0), `Bitmap "partial-mask-e";
     (0, 1), `Bitmap "partial-mask-s"] 
  in
  let masks () = List.concat (
    List.map 
      (fun (ofs, mask) ->
	 match get (offset +| ofs) with
	   | (Tile.Grass, _) -> [mask]
	   | _ -> []
      )
      masks
  )
  in
  let description = 
    let color = color_map (tile, info) in
      match tile with 
	| Tile.Space -> 
	    let ms = masks () in
	      if ms = [] then
		space_tile ()
	      else
		`Combine (`Move ((0, 20), `Compose ms),
			  space_tile (),
			  grass_tile ())
	| Tile.Grass -> 
	    `Compose [grass_tile (); `Realpha (float info.Tile.hp /. 10.0, `Bitmap "grass")];
	| Tile.Mountain ->
	    let ms = masks () in
	      if ms = [] then
		`Compose [space_tile ();
			  `Realpha (scale 0 50 info.Tile.hp,
				    `Move ((0, 20), `Bitmap "mountain"))]
	      else
		`Compose [`Combine (`Move ((0, 20), `Compose ms),
				    space_tile (),
				    grass_tile ());
			  `Realpha (scale 0 50 info.Tile.hp,
				    `Move ((0, 20), `Bitmap "mountain"))]
	| _ -> `Compose [space_tile (); `Move ((0, 10), `Char (Tile.char_of_tile tile, color))]
	    in
	      Bitmaps.bitmap description

let within_map map (x, y) =
  x >= 0 && y >= 0 && x < map.width && y < map.height

let at_full map (x, y) =
  if within_map map (x, y) then
    map.contents.(x + map.width * y)
  else
    Tile.tile_with_info Tile.Wall

let at map coord =
  fst (at_full map coord)

let bitmap_at map (x, y) =
  match map.bitmaps.(x + map.width * y) with
    | None -> 
	let t = create_tile (x, y) (at_full map) in
	  map.bitmaps.(x + map.width * y) <- Some t;
	  t
    | Some x -> x

let set map (x, y) value =
  map.contents.(x + map.width * y) <- value;
  map.bitmaps.(x + map.width * y) <- None

let damage map ((x, y) as at) damage_hp =
  if not (within_map map (x, y)) then
    0
  else
    let ch, info = at_full map at in
    let real_damage =
      match ch with
	| Tile.Grass 
	| Tile.Mountain -> 
	    damage_hp
	| _ -> 0
    in
    let { Tile.hp = hp } = info in
    let new_damage = max 0 (info.Tile.hp - real_damage) in
    let delta = hp - new_damage in
      set map at (ch, { Tile.hp = new_damage });
      delta
    
let tile_at map map_at = 
  if not (within_map map map_at) then
    Bitmaps.bitmap (`Move ((0, 10), `Char ('#', white)))
  else
    bitmap_at map map_at
	      
let create 
    ?info
    { MapData.size = width, height; 
      tile_size = tile_size; 
      data = contents }  =
  let contents = 
    (match info with
       | None -> Array.map Tile.tile_with_info contents
       | Some aux -> Array.mapi (fun i v -> (v, aux.(i))) contents);
  in
    { width	= width;
      height	= height;
      contents  = contents;
      tile_size = tile_size;
      bitmaps	= Array.map (fun _ -> None) contents
    }

let block_cmask =
  lazy (let bitmap = Bitmaps.bitmap (`Block (20, 20, (255, 255, 255), 255)) in
	Bitmaps.collision_mask_of_surface (snd bitmap))

class map_object map map_at tile_size at (collidable : bool) (destructible : bool) (friction : float) = 
object
  method get_state : float -> Sexplib.Sexp.t = failwith "Cannot get_state map_object"
  method set_state : Sexplib.Sexp.t -> float -> unit = failwith "Cannot set_state map_object"
  method self_owned = true
  method get_id = Object.MapId map_at
  method collision_area = 
    let ofs = (0.0, snd tile_size) in
      (at +|. ofs, at +|. ofs +|. tile_size), Lazy.force block_cmask
  method location = at
  method z_location = 0.0
  method render (_gfx : Gfx.t) (_:Object.render_flags) (_at : int * int) = ()
  method impact (world : Object.world) (damage_hp : int) : unit = 
    let got_damage = damage map map_at damage_hp in
      world#report_damage (Object.MapId map_at, got_damage);
    
  method collide (world : Object.world) = collidable

  method speed = 0.0, 0.0
  method set_speed (speed : float * float) = ()
  method collidable = collidable
  method destructible = destructible

  method friction = friction

  method cause_damage (_:int) = ()

  method is_alive = true

  method change = ()

  method tick (_:float) (_:Object.world) = ()

    (* temporary hack *)
  method set_self_owned (_:bool) = ()

  method set_location (at' : float * float) : unit = failwith "attempted to set a map tile coordinate"

  method changed_since (_:Object.serial) = false

  method test (_:Object.world) = ()

  method recharge = false
end

let real_coords map at =
  Vector.float2 (at *| map.tile_size)

let map_coords map at =
  at /|. Vector.float2 map.tile_size

let objects_near map ((x1, y1), (x2, y2)) : Object.t list =
  let objs = ref [] in
  let x1, y1 = map_coords map (x1, y1) in
  let x2, y2 = map_coords map (x2, y2) in
    for yc = (int_of_float (y1 -. 1.5)) to (int_of_float (y2 +. 0.0)) do
      for xc = (int_of_float (x1 -. 1.5)) to (int_of_float (x2 +. 0.0)) do
	let ch, info = 
	  if xc >= 0 && yc >= 0 && xc < map.width && yc < map.height then
	    at_full map(xc, yc)
	  else
	    Tile.tile_with_info (Tile.Wall)
	in
	let friction =
	  let hp = info.Tile.hp in
	  match ch with
	    | Tile.Grass when hp >= 5 -> 0.37
	    | Tile.Grass when hp >= 1 -> 0.15
	    | Tile.Grass -> 0.0
	    | Tile.Mountain -> 0.4
	    | _ -> 0.0
	in
	let collidable, destructible =
	  let hp = info.Tile.hp in
	  match ch with
	    | Tile.Mountain when hp > 50 -> true, true
	    | Tile.Mountain -> false, true
	    | Tile.Wall -> true, true
	    | Tile.Grass -> false, true
	    | _ -> false, false
	in
	  if info.Tile.hp > 0 && ch <> Tile.Space then
	    let obj = new map_object map (xc, yc) (Vector.float2 map.tile_size) (real_coords map (xc, yc)) collidable destructible friction in
	      objs := (obj :> Object.t) :: !objs
      done
    done;
    !objs

let ord_partition f l =
  let rec aux tail head =
    match tail with
      | x::xs when f x ->
	  aux xs (x::head)
      | _ ->
	  List.rev head, tail
  in
    aux l []

let render_map map (gfx:#Gfx.t) (flags:Object.render_flags) display_at (offset, size) (objects:#Object.t list) =
  gfx#with_clip_rect 
    (Sdlvideo.rect 
       (fst display_at) (snd display_at) 
       (int_of_float (fst size)) (int_of_float (snd size)))
    (fun () ->
       let map_offset = map_coords map offset in
       let map_size = Vector.int2 (size /|. Vector.float2 map.tile_size) in
       let offset_frac = 
	 let frac v = fst (modf v) in
	   frac (fst map_offset), frac (snd map_offset)
       in
       let depth (y, z) = y -. z in
       let objs =
	 List.sort
	   (fun (y1, _) (y2, _) -> compare y1 y2)
	   (List.map (fun o -> let (((_, _), (_, y2)), _) = o#collision_area in 
			(* why add z_location here? we want to
			   consider this object to be more south than
			   it actually is, for it to be rendered later
			*)
		      let render () =
			o#render (gfx :> Gfx.t) flags
			  (int_of_float2 
			     ((o#location -|. offset) +|. float2 (display_at))
			  )
		      in
			(depth (y2, o#z_location), render)
		     )
	      objects)
       in
       let map_objs =
	 GobaUtils.foldr2d 
	   (fun (x, y) objs ->
	      let at_y = snd offset +. (float y -. snd offset_frac) *. float (snd map.tile_size) in
	      let render () =
		let coord = (x, y) in
		let at = 
		  Vector.int2 
		    ((Vector.float2 coord -|. offset_frac) *|. Vector.float2 map.tile_size) in
		let map_at = coord +| Vector.int2 map_offset in
		let (surface_offset, surface) = tile_at map map_at in
		  gfx#blit (at +| surface_offset +| display_at) surface
	      in
		(depth (at_y +. 20.0, 0.0), render)::objs
	   )
	   []
	   ((-2, -2), (fst map_size + 2, snd map_size + 3))
       in
       let render_objs objs =
	 List.iter (fun (_, obj) -> 
		      obj ()
		   ) objs
       in
	 render_objs (List.merge (fun (y1, _) (y2, _) -> compare y1 y2) objs map_objs);
(*	 render_objs objs;*)
    )
    
