open VecOps

let gfx = new Gfx.t ~size:(800, 600) ()

let timer_hz = 120.0
let refresh_hz = try float_of_string (Sys.getenv "HZ") with _ -> 30.0
let world_tick_hz = 60.0

let timer_interval = 1.0 /. timer_hz
let refresh_interval = 1.0 /. refresh_hz
let world_tick_interval = 1.0 /. world_tick_hz

let () = 
  Sdlmixer.open_audio ();
  ignore (Sdlmixer.allocate_channels 16);
  if false then
    let music = Sdlmixer.load_music "goba.ogg" in
      Sdlmixer.play_music ~loops:0 music


let pending_mutex = Mutex.create ()
let pending_timers = ref 0

let rec ticker () =
  let extra = float !pending_timers *. 0.01 in
    (try ignore (Unix.select [] [] [] (1.0 /. refresh_hz +. extra)) with exn -> ());
    Sdlevent.add [Sdlevent.USER 0];
    Mutex.lock pending_mutex; incr pending_timers; Mutex.unlock pending_mutex;
    ticker ()

let tank_keys = [Sdlkey.KEY_1, Game.TankId 0;
		 Sdlkey.KEY_2, Game.TankId 1;
		 Sdlkey.KEY_3, Game.TankId 2;
		 Sdlkey.KEY_4, Game.TankId 3]

let firedirs = [Sdlkey.KEY_q, 360 - 90 - 45;
		Sdlkey.KEY_w, 360 - 90;
		Sdlkey.KEY_e, 360 - 45;

		Sdlkey.KEY_a, 180;
		Sdlkey.KEY_s, 90;
		Sdlkey.KEY_d, 0;

		Sdlkey.KEY_z, 90 + 45;
		Sdlkey.KEY_x, 90;
		Sdlkey.KEY_c, 45]

let dirs = [Sdlkey.KEY_DOWN, (fun delta game tank -> Game.accel_tank game tank (-. (150.0 *. delta)));
	    Sdlkey.KEY_UP, (fun delta game tank -> Game.accel_tank game tank (150.0 *. delta));
	    Sdlkey.KEY_LEFT, (fun delta game tank -> Game.turn_tank game tank (90.0 *. -.pi /. 180.0 *. delta));
	    Sdlkey.KEY_RIGHT, (fun delta game tank -> Game.turn_tank game tank (90.0 *. pi /. 180.0 *. delta));
	   ]

let avg_dir d1 d2 =
  let d1 = mod_float d1 pi2 in
  let d2 = mod_float d2 pi2 in
  let d2 = if d2 < d1 then d2 +. pi2 else d2 in
  let d1 = 
    if d1 < d2 -. pi then 
      d1 +. pi2 
    else
      d1
  in
    (d1 +. d2) /. 2.0

let run_menu ?(user_handler=fun continue _ -> continue ()) menu callback =
  let rec loop prev_time = 
    match Sdlevent.wait_event () with
      | Sdlevent.QUIT -> ()
      | Sdlevent.KEYDOWN { Sdlevent.keysym = Sdlkey.KEY_DOWN } ->
	  Menu.choose_next menu;
	  loop prev_time
      | Sdlevent.KEYDOWN { Sdlevent.keysym = Sdlkey.KEY_UP } ->
	  Menu.choose_prev menu;
	  loop prev_time
      | Sdlevent.KEYDOWN 
	  { Sdlevent.keysym = (Sdlkey.KEY_SPACE | Sdlkey.KEY_RETURN) } ->
	  callback (fun () -> loop prev_time) (Menu.choose menu)
      | Sdlevent.USER 0 ->
	  let () = 
	    Mutex.lock pending_mutex; 
	    let _v = !pending_timers in
	      decr pending_timers; 
	      Mutex.unlock pending_mutex
	  in
	    user_handler
	      (fun () -> 
		 let now = Unix.gettimeofday () in
		 let delta = now -. prev_time in
		   Menu.tick menu delta;
		   gfx#clear ();
		   Menu.render menu gfx;
		   gfx#sync ();
		   loop now)
	      0
      | Sdlevent.USER n ->
	  user_handler (fun () -> loop prev_time) n
      | _ -> loop prev_time
  in
    loop (Unix.gettimeofday ())
      
let rec player_menu client game_loop =
  let unhook () = 
    Client.set_available_players_handler client Client.nothing
  in
    Client.set_available_players_handler client
      (fun available ->
	 Sdlevent.add [Sdlevent.USER 1]
      );
    let available = Client.available client in
    let player n =
      Menu.entry (Printf.sprintf "Player %d" (n + 1)) n
    in
      run_menu 
	~user_handler:(fun continue n ->
			 if n = 1 then
			   player_menu client game_loop
			 else
			   begin
			     Client.handle_io client;
			     continue ()
			   end
		      )
	(Menu.create (List.map player (
			List.filter 
			  (fun id -> List.mem (Player.Id id) available)
			  [0; 1; 2; 3]
		      )))
	(fun _ player ->
	   unhook ();
	   game_loop client player
	)

let connect_menu game_loop =
  run_menu
    (Menu.create [Menu.entry "Connect localhost" (Some "localhost");
		  Menu.entry "Connect jolt" (Some "jolt.modeemi.fi");
		  Menu.entry "Connect.." None])
    (fun _ server ->
       let connect server =
	 let host = Unix.gethostbyname server in
	 let server = (Unix.ADDR_INET (host.Unix.h_addr_list.(0), 61753)) in
	 let client = Client.create server in
	   player_menu client game_loop
       in
	 match server with
	   | Some server -> connect server
	   | None -> failwith "not supported: use command line (first argument is the server to connect to)"
    )


let main_menu game_loop =
  run_menu 
    (Menu.create [Menu.entry "Join game" `Start;
		  Menu.entry "Start server" `None;
		  Menu.entry "Quit" `Quit])
    (fun continue chosen ->
       match chosen with
	 | `Start -> connect_menu game_loop
	 | `Quit -> ()
	 | `None -> continue ()
    )

type game_state = { cur_tanks		: Game.tank_id list;
		    world_time		: float;
		    world_changed	: bool;
		    refresh_time	: float;
		    chat		: ReadInput.t option; }

let game_loop server player_id =
  let frame = ref 0 in
  let game = Game.create server player_id in
  let rec loop prev_time state =
    match Sdlevent.wait_event () with
      | Sdlevent.QUIT -> 
	  Printf.printf "Quit\n%!";
      | Sdlevent.KEYDOWN { Sdlevent.keysym = Sdlkey.KEY_RETURN } when state.chat = None ->
	  loop prev_time { state with chat = Some (ReadInput.create (0, 420)) }
      | (Sdlevent.KEYDOWN _) as event when state.chat <> None ->
	  begin
	    match state.chat with
	      | None -> loop prev_time state
	      | Some chat ->
		  match ReadInput.handle chat event with
		    | Some msg -> 
			Game.chat game msg;
			loop prev_time { state with chat = None }
		    | None -> 
			loop prev_time state
	  end;
      | Sdlevent.KEYDOWN { Sdlevent.keysym = key; keymod = mods } when List.exists (fun (key', _) -> key = key') tank_keys ->
	  let state = 
	    { state with cur_tanks =
		List.fold_left 
		  (fun cur_tanks (key', tank) ->
		     if key' = key then
		       if mods land Sdlkey.kmod_shift <> 0 then
			 if List.exists ((=) tank) cur_tanks then
			   List.filter ((<>) tank) cur_tanks
			 else
			   tank::cur_tanks
		       else
			 [tank]
		     else
		       cur_tanks
		  )
		  state.cur_tanks
		  tank_keys
	    }
	  in
	    loop prev_time state
      | Sdlevent.USER 0 ->
	  let () = 
	    Mutex.lock pending_mutex; 
	    let _v = !pending_timers in
	      decr pending_timers; 
	      Mutex.unlock pending_mutex
	  in
	  let handle_controls delta =
	    let projectile_speed = 300.0 in
	      if Sdlkey.is_key_pressed Sdlkey.KEY_m then
		List.iter (Game.drop_mine game) state.cur_tanks;
	      if not (List.fold_left 
			(fun was (key, f) ->
			   if Sdlkey.is_key_pressed key then
			     begin
			       List.iter
				 (fun tank ->
(*				    Game.accel_tank game tank (vec *|. Vector.dup (accel *. delta))*)
				    f delta game tank;
				 )
				 state.cur_tanks;
			       true
			     end
			   else
			     was)
			false
			dirs) then
		List.iter 
		  (fun tank ->
		     (match Game.tank game tank with None -> () | Some tank -> tank#decelerate (1.0 ** delta)))
		  state.cur_tanks;
	      let fire_dir = 
		let dirs = 
		  List.map (fun (_, dir) -> float dir /. 180.0 *. 4.0 *. atan 1.0) (
		    List.filter (fun (key, dir) -> Sdlkey.is_key_pressed key) firedirs
		  )
		in
		  if dirs <> [] then
		    let min_dir = List.fold_left min (List.hd dirs) (List.tl dirs) in
		    let max_dir = List.fold_left max (List.hd dirs) (List.tl dirs) in
		    let dir = avg_dir min_dir max_dir +. (Random.float 1.0 -. 0.5) /. pi2 in
		      Some (cos dir, sin dir)
		  else
		    None
	      in
		begin
		  match fire_dir with
		    | None -> ()
		    | Some dir -> List.iter (fun tank -> Game.fire game tank (dir *|. Vector.dup projectile_speed)) state.cur_tanks;
		end;
	  in

	  let step delta =
	    if state.chat = None then
	      handle_controls delta;
	    Game.tick game delta;
	  in
	  let now = Unix.gettimeofday () in
	  let delta = now -. prev_time in
	  let rec ticktick (changed, left) =
	    if left >= world_tick_interval then
	      begin
		step world_tick_interval;
		ticktick (true, left -. world_tick_interval)
	      end
	    else
	      changed, left
	  in
	    Game.handle_io game;
	    let world_changed, world_time = ticktick (state.world_changed, state.world_time +. delta) in
	    let world_changed, refresh_time =
	      let t = state.refresh_time +. delta in
		if t >= refresh_interval && world_changed then
		  begin
		    (*Printf.printf "frame %d\n%!" !frame;*)
		    incr frame;
		    gfx#clear ();
		    Game.render game gfx (state.cur_tanks);
		    (match state.chat with Some chat -> ReadInput.render chat gfx | None -> ());
		    if Sdlkey.is_key_pressed Sdlkey.KEY_F1 then
		      Help.render gfx;
		    gfx#sync ();
		    false, t -. refresh_interval
		  end
		else
		  world_changed, t
	    in
	    let state = 
	      { state with 
		  world_time = world_time;
		  world_changed = world_changed;
		  refresh_time = refresh_time } in
	      loop now state;
      | _ -> loop prev_time state
  in
    loop (Unix.gettimeofday ()) { cur_tanks = [Game.TankId 0];
				  world_time = 0.0;
				  world_changed = false;
				  refresh_time = 0.0;
				  chat = None; }

let main args =
  let _ = Thread.create ticker () in
    begin
      match args with
	| server::_ ->
	    let host = Unix.gethostbyname server in
	    let server = (Unix.ADDR_INET (host.Unix.h_addr_list.(0), 61753)) in
	    let client = Client.create server in
	      player_menu client game_loop
	| _ -> main_menu game_loop;
    end;
    Printf.printf "Bye!\n%!";
    Perf.show_counters ()

let imain () =
  ()

let _ = 
  if !Sys.interactive then 
    imain ()
  else
    begin
      Random.self_init ();
      main (List.tl (Array.to_list Sys.argv))
    end
