type vector = float * float

let (+.|) (x1, y1) (x2, y2) =
  (x1 +. x2, y1 +. y2)

let (-.|) (x1, y1) (x2, y2) =
  (x1 -. x2, y1 -. y2)

let (+|) (x1, y1) (x2, y2) =
  (x1 + x2, y1 + y2)

let (-|) (x1, y1) (x2, y2) =
  (x1 - x2, y1 - y2)

let (/.|) (x, y) v =
  (x /. v, y /. v)

let ( *.| ) (x, y) v =
  (x *. v, y *. v)

let dot (x1, y1) (x2, y2) =
  x1 *. x2 +. y1 *. y2

let inside ((x0, y0), (x1, y1)) (x, y) =
  x >= x0 && x <= x1 && y >= y0 && y <= y1

let pow2 x = x *. x

let distance (x, y) =
  sqrt (pow2 x +. pow2 y)

let coaxel (x, y) =
  (0.0 -. y, x)

let base (x, y) = 
  ((x, y), coaxel (x, y))

let inverse ((a, b), (c, d)) =
  let v = 1.0 /. (a *. d -. b *. c) in
    ((v *. d, -.v *. b), (-.v *. c, v *. a))

let ( *.|| ) (x, y) ((a, b), (c, d)) =
  ( x *. a +. y *. b ), (x *. c +. y *. d)

let float2 (x, y) =
  float x, float y

let int_of_float2 (x, y) =
  int_of_float x, int_of_float y

let format_float2 (x, y) ch =
  Format.fprintf ch "(%f,%f)" x y

let collide at1 at2 radius speed =
  let delta = (at2 -.| at1) in
  distance delta <= radius &&
    match speed with
      | None -> true
      | Some speed -> dot speed delta < 0.0

let distance' delta speed ratio =
  let base = base speed in
  let delta = delta *.|| inverse base *.|| ((1.0, 0.0), (0.0, ratio)) *.|| base in
    distance delta

let collide' at1 at2 radius speed ratio =
  let delta = (at2 -.| at1) in
  distance' delta speed ratio <= radius 
  && dot speed delta < 0.0


(* ax + by = c *)
let line_equation (x1, y1) (x2, y2) =
  assert ((x1, y1) <> (x2, y2));
  let xd = x2 -. x1 in
  let yd = y2 -. y1 in
  if abs_float xd > abs_float yd then
    let xslope = yd /. xd in
      (* y = xslope * x + b where y = y1, x = x1*)
    let b = y1 -. xslope *. x1 in
      (* -xslope * x + y = b *)
      -.xslope, 1.0, b
  else
    let yslope = xd /. yd in
      (* y = xslope * x + b where y = y1, x = x1*)
    let b = x1 -. yslope *. y1 in
      (* x + -yslope * y = b *)
      1.0, -.yslope, b

(*
maxima:
(%i27) solve([x^2+y^2=r^2,a*x+b*y=c], [x, y]);
                        2    2   2    2
               b sqrt((b  + a ) r  - c ) - a c
(%o27) [[x = - -------------------------------,
                            2    2
                           b  + a
            2  2    2  2    2
    a sqrt(b  r  + a  r  - c ) + b c
y = --------------------------------],
                 2    2
                b  + a
              2    2   2    2                           2  2    2  2    2
     b sqrt((b  + a ) r  - c ) + a c      b c - a sqrt(b  r  + a  r  - c )
[x = -------------------------------, y = --------------------------------]]
                  2    2                               2    2
                 b  + a                               b  + a
*)

let pi = 4.0 *. atan 1.0

(*
let intersect_line_circle 
    (a, b)
    (at, r) = 
  let (a, b, c) = line_equation (a -.| at) (b -.| at) in
  let p1 = 
    ~-.((b *. sqrt((pow2 b +. pow2 a) *. pow2 r -. pow2 c) -. a *. c) /. (pow2 b +. pow2 a)),
    (a *. sqrt(pow2 b *. pow2 r +. pow2 a *. pow2 r -. pow2 c) +. b *. c) /. (pow2 b +. pow2 a) in
  let p2 =
    (b *. sqrt((pow2 b +. pow2 a) *. pow2 r -. pow2 c) +. a *. c) /. (pow2 b +. pow2 a),
    (b *. c -. a *. sqrt(pow2 b *. pow2 r +. pow2 a *. pow2 r -. pow2 c)) /. (pow2 b +. pow2 a) 
  in
    (p1 +.| at, p2 +.| at)
*)

let intersect_line_circle
    (a, b)
    (at, r) = 
  let (a, b, c) = line_equation (a -.| at) (b -.| at) in
  let c2 = (pow2 c) in
  let a2 = (pow2 a) in
  let b2 = (pow2 b) in
  let r2 = (pow2 r) in
  let b2a2 = (b2 +. a2) in
  let eq1 = b2a2 *. r2 -. c2 in
  let eq2 = b2 *. r2 +. a2 *. r2 -. c2 in
  let eq3 = b2a2 *. r2 -. c2 in
  let eq4 = b2 *. r2 +. a2 *. r2 -. c2 in
    if (eq1 >= 0.0 &&
	   eq2 >= 0.0 &&
	   eq3 >= 0.0 &&
	   eq4 >= 0.0 &&
	   b2a2 > 0.0) then
      Some 
	(let p1 = 
	  ~-.((b *. sqrt(eq1) -. a *. c) /. b2a2),
	  (a *. sqrt(eq2) +. b *. c) /. b2a2 in
	let p2 =
	  (b *. sqrt(eq3) +. a *. c) /. b2a2,
	  (b *. c -. a *. sqrt(eq4)) /. b2a2 
	in
	  (p1 +.| at, p2 +.| at))
    else
      None
	  
let faces v1 v2 = dot v1 v2 > 0.0
 
let reflect (at, speed) (mat, mradius) =
  let base = base (coaxel (mat -.| at)) in
  let x, y = speed *.|| base in
  let speed' = (x, ~-.y) *.|| inverse base in
    at, speed'
