let cursor_of_string (w, h) (hx, hy) s =
  let data = Array.init (w * h) 
    (fun i -> if s.[i] = 'X' then 1 else 0)
  in
  let mask = Array.init (w * h) 
    (fun i -> if s.[i] = '-' then 0 else 1)
  in
    Sdlmouse.convert_to_cursor data mask w h hx hy

let default_font_size = 40

let crosshair () =
  cursor_of_string (2, 2) (0, 0)
  "
-----------#-#####-#-----------
----------#X##X#X##X#----------
-----------#X#X#X#X#-----------
------------#XX#XX#------------
-------------#X#X#-------------
-------------#X#X#-------------
-------------#X#X#-------------
-------------#X#X#-------------
-------------#X#X#-------------
-------------#X#X#-------------
-#---------#-#X#X#-#---------#-
#X#-------#X##X#X##X#-------#X#
-#X#-------#X#X#X#X#-------#X#-
###X#########XX#XX#########X###
#XXXXXXXXXXXXXX#XXXXXXXXXXXXXX#
###############################
#XXXXXXXXXXXXXX#XXXXXXXXXXXXXX#
###X#########XX#XX#########X##-
-#X#-------#X#X#X#X#-------#X#-
#X#-------#X##X#X##X#-------#X#
-#---------#-#X#X#-#---------#-
-------------#X#X#-------------
-------------#X#X#-------------
-------------#X#X#-------------
-------------#X#X#-------------
-------------#X#X#-------------
-------------#X#X#-------------
------------#XX#XX#------------
-----------#X#X#X#X#-----------
----------#X##X#X##X#----------
-----------#-#####-#-----------
"

(* create default_cursor during the first call *)
let default_cursor =
  let c = ref None in
    fun () ->
      if !c = None then
	c := Some (Sdlmouse.get_cursor ());
      match !c with Some x -> x | None -> assert false

let font = 
  let fonts = Hashtbl.create 10 in
    fun size -> 
      try Hashtbl.find fonts size 
      with Not_found ->
	let f = Sdlttf.open_font "font.ttf" size in
	  Hashtbl.add fonts size f;
	  f

let get_font ?(size=default_font_size) () = font size
	
class t ?(size=(800, 600)) () = 
  let _ = Sdl.init [`EVERYTHING] in
  let _ = Sdlttf.init () in
  let fs = List.exists ((=) "-fs") (Array.to_list Sys.argv)in
  let flags = 
    if fs then
      [`DOUBLEBUF; `ANYFORMAT; `SWSURFACE; `FULLSCREEN] else 
      [`DOUBLEBUF; `ANYFORMAT; `HWSURFACE]
  in
  let surface = Sdlvideo.set_video_mode (fst size) (snd size) flags in
  let () =
    Printf.printf "Driver: %s\n%!" (Sdlvideo.driver_name ());
    Printf.printf "Flags: %s\n%!"
      (String.concat " " 
	  (List.map 
	      (function
		| `HWSURFACE -> "hwsurface"
		| `SWSURFACE -> "swsurface"
		| `ANYFORMAT -> "anyformat"
		| `DOUBLEBUF -> "doublebuf"
		| `FULLSCREEN -> "fullscreen"
		| `HWACCEL -> "hwaccel"
		| `SRCCOLORKEY -> "srccolorkey"
		| `RLEACCEL -> "rleaccel"
		| `SRCALPHA -> "srcalpha"
		| `PREALLOC -> "prealloc"
		| _ -> "some"
	      ) (Sdlvideo.surface_flags surface)
	  )
      )
  in
  let _ = Sdlkey.enable_key_repeat () in
  let _ = Sdlkey.enable_unicode true in
  let _ = default_cursor () in
object
  method write ?(color=(255,255,255)) ?(size=default_font_size) (x, y) str = 
    let msg = Sdlttf.render_text_blended (font size) str color in
    let si = Sdlvideo.surface_info msg in
      Sdlvideo.blit_surface ~src:msg ~dst:surface ~dst_rect:(Sdlvideo.rect ~x ~y ~w:si.Sdlvideo.w ~h:si.Sdlvideo.h) ()

  method show_cursor visible =
    Sdlmouse.show_cursor visible

  method set_cursor cursor =
    Sdlmouse.set_cursor cursor

  method get_mouse () =
    Sdlmouse.get_state ()

  method text_surface ?(color=(255, 255, 255)) ?(size=default_font_size) str = 
    Sdlvideo.display_format ~alpha:true (Sdlttf.render_text_blended (font size) str color)

  method clear () =
    Sdlvideo.fill_rect surface 0l

  method sync ?rect () = Sdlvideo.update_rect ?rect surface

  method flip () = Sdlvideo.flip surface
    
  method blit ?(sync=false) (x, y) bitmap =
    let si = Sdlvideo.surface_info bitmap in
    let rect = Sdlvideo.rect ~x ~y ~w:si.Sdlvideo.w ~h:si.Sdlvideo.h in
      Sdlvideo.blit_surface ~src:bitmap ~dst:surface ~dst_rect:rect ();
      if sync then Sdlvideo.update_rect ~rect surface

  method formatted ?alpha surface =
    Sdlvideo.display_format ?alpha surface

  method create_surface flags (w, h) = Sdlvideo.create_RGB_surface_format surface flags ~w ~h

  method grab (x, y) (w, h) =
    let s = Sdlvideo.create_RGB_surface_format surface [] ~w ~h in
      Sdlvideo.blit_surface ~src:surface ~dst:s ~src_rect:(Sdlvideo.rect ~x ~y ~w ~h) ();
      s

  method map_rgb ?alpha (r, g, b) = 
    Sdlvideo.map_RGB surface ?alpha (r, g, b)

  method put_pixel (x, y) color =
    Sdlvideo.put_pixel surface x y color

  method with_clip_rect : 'a. ('rect -> (unit -> 'a) -> 'a) = fun r f ->
    let old_clip = Sdlvideo.get_clip_rect surface in
      Sdlvideo.set_clip_rect surface r;
      let v = 
	(try 
	    let v = f () in
	      (fun () -> v)
	  with
	    | exn ->
		fun () -> raise exn) in
	Sdlvideo.set_clip_rect surface old_clip;
	v ()

  method rectangle ?(sync=false) ?alpha color (x, y) (w, h) =
    let rect = Sdlvideo.rect ~x ~y ~w ~h in
    Sdlvideo.fill_rect ~rect surface (Sdlvideo.map_RGB surface ?alpha color);
      if sync then Sdlvideo.update_rect ~rect surface

  method nonfilled_rectangle ?(sync=false) color ?alpha (x, y) (w, h) =
    let c = Sdlvideo.map_RGB surface ?alpha color in
    let f x y w h =
      Sdlvideo.fill_rect ~rect:(Sdlvideo.rect ~x ~y ~w ~h) surface c;
    in
      f x y w 1;
      f x (y + h - 1) w 1;
      f x (y + 1) 1 (h - 2);
      f (x + w - 1) (y + 1) 1 (h - 2);
      if sync then Sdlvideo.update_rect ~rect:(Sdlvideo.rect ~x ~y ~w ~h) surface

  method get_surface = surface
end

let surface_dims s =
  let { Sdlvideo.w = w; h = h } = Sdlvideo.surface_info s in
    w, h

let dup_surface flags from =
  let (w, h) = surface_dims from in
  let surface = Sdlvideo.create_RGB_surface_format from flags ~w ~h in
    Sdlvideo.blit_surface ~src:from ~dst:surface ();
    surface
    
