type layer = int

type mid = int64

let layer n = n

let default_layer = 0

module LayerMap = struct
  include Map.Make (
    struct 
      type t = layer
      let compare a b = compare a b
    end
    )
    
  let fold f map = Utils.fold_map (fun f -> iter f map) f
end

type 'a mc = ('a -> mcc list)
and 'a mcv =  
    { mcv_id : mid;
      mcv_layer : layer;
      mcv_mc : 'a mc }
and mcc =
    | MC_SetClick of mc_click mcv
    | MC_SetTick of mc_tick mcv
    | MC_SetMove of mc_move mcv
    | MC_Renew
    | MC_Unset of mid
and mc_tick = float
and mc_click = (int * int)
and mc_move = (int * int)
and mct = { mcr_ticks : mc_tick mcv list LayerMap.t;
	    mcr_clicks : mc_click mcv list LayerMap.t;
	    mcr_moves : mc_move mcv list LayerMap.t;
	  }

let append map element =
  let l = 
    try LayerMap.find element.mcv_layer map
    with Not_found -> []
  in
    (*Printf.printf "Adding stuff to layer %d\n%!" layer;*)
    LayerMap.add element.mcv_layer (element::l) map

let new_mid =
  let c = ref 0L in
    fun () ->
      c := Int64.add !c 1L;
      !c

let maybe_new_mid id =
  match id with
    | None -> new_mid ()
    | Some x -> x

let create () =
  { mcr_ticks = LayerMap.empty;
    mcr_clicks = LayerMap.empty;
    mcr_moves = LayerMap.empty;
  }

let remove_id m (id : mid) =
  LayerMap.map
    (fun els ->
      List.filter
	(fun { mcv_id = id' } ->
	  id <> id')
	els
    )
    m
    

let handle_requests mct (cmds : mcc list) = 
  List.fold_left
    (fun mct ->
      function
	| MC_SetClick handler ->
	    { mct with mcr_clicks = append mct.mcr_clicks handler }
	| MC_SetTick handler ->
	    { mct with mcr_ticks = append mct.mcr_ticks handler }
	| MC_SetMove handler ->
	    { mct with mcr_moves = append mct.mcr_moves handler }
	| MC_Renew ->
	    (* handled earlier *)
	    assert false
	| MC_Unset id ->
	    { mct with 
	      mcr_clicks = remove_id mct.mcr_clicks id;
	      mcr_ticks = remove_id mct.mcr_ticks id;
	      mcr_moves = remove_id mct.mcr_moves id;
	    }
    )
    mct
    cmds

let set_click ?(layer = default_layer) ?mid f = 
  MC_SetClick { mcv_layer = layer;
		mcv_id = maybe_new_mid mid;
		mcv_mc = f }
    
let set_tick ?(layer = default_layer) ?mid f = 
  (*Printf.printf "Creating tick for layer %d\n%!" layer;*)
  MC_SetTick { mcv_layer = layer;
	       mcv_id = maybe_new_mid mid;
	       mcv_mc = f }

let set_move ?(layer = default_layer) ?mid f = 
  (*Printf.printf "Creating tick for layer %d\n%!" layer;*)
  MC_SetMove { mcv_layer = layer;
	       mcv_id = maybe_new_mid mid;
	       mcv_mc = f }

let handle ?(debug = false) (value : 'a) new_mct renew (mcs : 'a mcv list LayerMap.t) =
  if debug then Printf.printf "Begin handle\n";
  let cmds : mcc list =
    List.concat (
	LayerMap.fold
	  (fun layer controls l -> 
	    if debug then Printf.printf "Handling layer %d (%d controls)\n" layer (List.length controls);
	    List.concat (
		List.fold_left 
		  (fun l ({ mcv_mc = control } as mcv) -> 
		    let cs = control value in
		    let cs = List.map 
			(function
			  | MC_Renew -> renew mcv
			  | x -> x)
			cs
		    in
		      cs::l) [] controls)
	    ::l
	  )
	  mcs
	  []
      )
  in
  let v = handle_requests new_mct cmds in
    if debug then Printf.printf "End handle\n%!";
    v

let renew = MC_Renew
  
let renew_map (f : ?layer : layer -> ?mid : mid -> 'a -> 'b) c =
  f ~layer:c.mcv_layer ~mid:c.mcv_id c.mcv_mc

let handle_tick value mct = 
  handle value { mct with mcr_ticks = LayerMap.empty } (renew_map set_tick) mct.mcr_ticks

let handle_click value mct = 
  handle value { mct with mcr_clicks = LayerMap.empty } (renew_map set_click) mct.mcr_clicks

let handle_move value mct = 
  handle value { mct with mcr_moves = LayerMap.empty } (renew_map set_move) mct.mcr_moves
    
let unset : mid -> mcc =
  fun id -> MC_Unset id

let handle_mcs mcs mct =
  handle_requests mct (List.concat mcs)

