open VecOps

type tank_id = TankId of int

type 'a object_list = (Object.object_id, 'a) Hashtbl.t

let sync_interval = 1.0 /. 30.0

let tank_sound = Sfx.Sample "tankkiloop"

type t = { player_id	: Player.id;
	   map		: GobaMap.t;
	   my_tanks	: (tank_id, Object.object_id) Hashtbl.t;
	   mutable base	: Object.object_id;
	   mutable projectiles : Projectile.t list;
	   client	: Client.t;

	   mutable pending_ping : bool;
	   ping_value	: float option ref;

	   objects	: Object.base object_list;
	   tanks	: Tank.t object_list;
	   bases	: Base.t object_list;

	   latency	: float ref;

	   mutable last_sync : float;
	   mutable last_sync_serial : Object.serial;

	   mutable object_serial : int64;
	   object_increment	: int64;

	   messages	: (string * Sdlvideo.surface) list ref;
	 }

let tank_channels = Array.init 4 (fun _ -> new SfxSdl.channel (Sfx.alloc_channel ()))
let fire_sound = Sfx.Sample "ampu"
let fire_channel = new SfxSdl.channel (Sfx.alloc_channel ())

(* lot's of magic happens in this function!

better cut down the side effects of the protocol somehow.. 

start asynchronous receiving events after this initialization? needs
protocol changes.

*)
exception JoinFailed

let create client player_num =
  let tanks = Hashtbl.create 10 in
  let objects = Hashtbl.create 10 in
  let bases = Hashtbl.create 10 in
  let messages = ref [] in

  let latency = ref 0.0 in

  let handle_object { Protocol.oi_id = id;
		      oi_pid = owner;
		      oi_state = state;
		      oi_type = otype } delay =
    match id with
      | Object.MapId at -> failwith "Cannot handle map objects!"
      | Object.ObjectId id ->
	  let delay = delay +. !latency /. 2.0 in
	    if try ignore (Hashtbl.find objects id); false with Not_found -> true then
	      let register hash obj = Hashtbl.add hash (Object.to_object_id (obj#get_id)) obj in
	      let obj =
		match otype with
		  | ObjectType.Base ->
		      let obj = new Base.t id (Object.State (state, delay)) in
			register bases obj;
			(obj :> Object.base);
		  | ObjectType.Tank ->
		      let obj = new Tank.t id (Object.State (state, delay)) in
			register tanks obj;
			(obj :> Object.base);
		  | ObjectType.Projectile -> 
		      let obj = new Projectile.t id (Object.State (state, delay)) in
			(obj :> Object.base)
		  | ObjectType.Explosion ->
		      let obj = new Explosion.t id (Object.State (state, delay)) in
			(obj :> Object.base)
		  | ObjectType.Mine ->
		      let obj = new Mine.t id (Object.State (state, delay)) in
			(obj :> Object.base)
	      in
		register objects obj
	    else
	      let obj = Hashtbl.find objects id in
		obj#set_state state delay
  in
  let handle_destruct id =
    Hashtbl.remove objects (Object.to_object_id id)
  in
  let map = 
    lazy (let map, info = Client.map client in
	    GobaMap.create ~info map) 
  in
  let handle_chat (from, msg) =
    let msg = from ^ "> " ^ msg in
      if msg <> "" then
	begin
	  let surface = Sdlttf.render_text_blended (Gfx.get_font ~size:20 ()) msg (200, 255, 128) in
	    messages := (msg, surface) :: !messages;
	    if List.length !messages > 5 then
	      messages :=
		fst (
		  GobaUtils.foldl1d
		    (fun _ (new_msgs, msgs) ->
		       List.hd msgs::new_msgs, List.tl msgs
		    )
		    ([], !messages)
		    (0, 5)
		)
	end
  in
  let handle_damage_object (id, damage) =
    match id with
      | Object.MapId at ->
	  let tile, { Tile.hp = hp } = GobaMap.at_full (Lazy.force map) at in
	    GobaMap.set 
	      (Lazy.force map) at (tile, { Tile.hp = hp - damage })
      | Object.ObjectId id ->
	  match try Some (Hashtbl.find objects id) with Not_found -> None with
	    | Some obj ->
		(*Printf.printf "Causing damage to object %Ld\n%!" (Object.int_of_id id);*)
		obj#cause_damage damage
	    | None -> ()
  in
    Client.set_object_handler client handle_object;
    Client.set_destruct_handler client handle_destruct;
    Client.set_damage_object_handler client handle_damage_object;
    Client.set_chat_handler client handle_chat;
    let player_id, base_id, my_tanks, (obj_base, obj_increment) =
      let my_tanks = Hashtbl.create 4 in
      let player_id, obj_ids, obj_constr = 
	Printf.printf "Begin join\n%!";
	match Client.join client player_num (Unix.getlogin ()) with
	  | Some x -> x
	  | None -> raise JoinFailed
      in
      let rec wait_objs () =
	Printf.printf "Waiting %s\n%!" (String.concat " " (List.map (fun id -> Object.string_of_id (Object.ObjectId id)) obj_ids));
	Client.handle_io client;
	(try ignore(Unix.select [] [] [] 0.1) with exn -> ());
	if List.for_all (fun id -> Hashtbl.mem objects id) obj_ids then
	  ()
	else
	  wait_objs ()
      in
	Printf.printf "Begin wait\n%!";
	wait_objs ();
	Printf.printf "Wait done\n%!";
	ignore (List.fold_left 
		  (fun cur_id tank_id ->
		     Hashtbl.add my_tanks (TankId cur_id) tank_id;
		     let tank = Hashtbl.find tanks tank_id in
		       tank#set_self_owned true;
		       tank#set_ident (Printf.sprintf "%d" (cur_id + 1));
		       cur_id + 1
		  ) 0 (List.tl obj_ids));
	(Hashtbl.find objects (List.hd obj_ids))#set_self_owned true;
	player_id, (List.hd obj_ids), my_tanks, obj_constr
    in
      { player_id	= player_id;
	client		= client;
	map		= Lazy.force map;
	my_tanks	= my_tanks;
	projectiles	= [];
	base		= base_id;
	objects		= objects;
	tanks		= tanks;
	bases		= bases;
	last_sync	= 0.0;
	last_sync_serial = Object.no_serial;
	latency		= latency;
	object_serial	= obj_base;
	object_increment = obj_increment;
	pending_ping	= false;
	ping_value	= ref None;
	messages	= messages;
      }

let tank t id = 
  try Some (Hashtbl.find t.tanks (Hashtbl.find t.my_tanks id))
  with Not_found -> None

let base t = Hashtbl.find t.bases t.base 

let find_object t obj =
  Hashtbl.find t.objects obj

(*let objects ?(projectiles=true) t =
  ((base t :> Object.base) ::
    (my_tanks t 
     @ if projectiles then (t.projectiles :> Object.base list) else []
    ))*)

let my_tanks t =
  GobaUtils.list_of_hash (fun _ oid -> (find_object t oid :> Object.base)) t.my_tanks

let objects ?except ?collidables ?self_owned ?changed_since ?filter t =
  Hashtbl.fold (fun id obj l -> 
		  if 
		    (collidables = None || Some obj#collidable = collidables)
		      && (self_owned = None || Some obj#self_owned = self_owned)
		      && (match changed_since with 
			    | None -> true
			    | Some t -> obj#changed_since t)
		      && (match except with 
			    | None -> true
			    | Some obj' -> obj != obj')
		      && (match filter with
			    | None -> true
			    | Some filter -> filter obj)
		  then
		    obj::l
		  else
		    l) t.objects []

let accel_tank t id adjust =
  match tank t id with
    | None -> ()
    | Some tank ->
	tank#accelerate adjust;
	let (TankId tank_num) = id in
	let chan = tank_channels.(tank_num) in
	  if not chan#playing then
	    chan#play ~loops:~-1 tank_sound;
	  chan#expire (Some 0.5)

let turn_tank t id dir =
  match tank t id with
    | None -> ()
    | Some tank ->
	tank#turn dir

let new_object_id t =
  let id = Object.object_id_of_int64 t.object_serial in
    t.object_serial <- Int64.add t.object_serial t.object_increment;
    id

let object_info t o =
  let id, at, speed = o#get_id, o#location, o#speed in
    id, at +|. speed *|. Vector.dup (!(t.latency) /. 2.0), speed

let new_object_info t o otype : Protocol.object_info =
  let id = o#get_id in
    { Protocol.oi_id = id;
      oi_pid = t.player_id;
      oi_state = o#get_state (!(t.latency) /. 2.0);
      oi_type = otype }

let drop_mine t id =
  match tank t id with
    | None -> ()
    | Some tank ->
	if tank#signal_mine then
	  let m = 
	    new Mine.t (new_object_id t) (Object.Init { Mine.init_at = tank#location }) in
	    Client.create_object t.client (new_object_info t m ObjectType.Mine);
	    m#set_self_owned true;
	    Hashtbl.add t.objects (Object.to_object_id m#get_id) (m :> Object.base) 

let fire t id dir =
  match tank t id with
    | None -> ()
    | Some tank ->
	if tank#signal_firing then
	  (* shoot to the direction of the tank *)
	  let dir = 
	    let tank_face = 
	      let dir = -. tank#get_direction -. pi /. 2.0 in
		(cos dir, sin dir) in
	      dir **|. Vector.base tank_face in
	  let p = new Projectile.t (new_object_id t) 
	    (Object.Init 
	       { Projectile.init_at = tank#location +|. tank#turret_offset (atan2 (snd dir) (fst dir));
		 init_speed = (dir +|. tank#speed);
		 init_orientation = dir;
		 init_parent = tank#get_id } )
	  in
	    fire_channel#play fire_sound;
	    Client.create_object t.client (new_object_info t p ObjectType.Projectile);
	    p#set_self_owned true;
	    t.projectiles <- p :: t.projectiles;
	    Hashtbl.add t.objects (Object.to_object_id p#get_id) (p :> Object.base)

let tick t delta =
  let objs ?except collidables = objects ?except ~collidables t in
  let world collidables self = 
    let near_objs = lazy (GobaMap.objects_near t.map (fst self#collision_area)) in
    let objs = 
      lazy ( let os = ((objs ~except:self collidables) :> Object.t list)  in
	     let os = os @ Lazy.force near_objs in
	       (*		  let os = os @ (t.projectiles :> Object.t list) in*)
	       os
	   ) in
      (object
	 method explode =
	   (*	  (fun at -> ());*)
	  (fun at burst ->
	     let id = new_object_id t in
	     let obj = (new Explosion.t id (Object.Init { Explosion.init_at = at;
							  init_burst = burst }) :> Object.base) in
	       obj#set_self_owned true;
	       Client.create_object t.client 
		 (new_object_info t obj ObjectType.Explosion);
	       Hashtbl.add t.objects id obj
	  )
	 method collision = 
	   (fun obj filter -> 
	      Collision.collisions2 (obj : #Collision.t) (Lazy.force objs) filter
	   );
	 method env =
	   (fun obj filter ->
	      Collision.collisions (obj : #Collision.t) (Lazy.force near_objs) filter);
	 method objs =
	   (fun ((x1, y1), (x2, y2)) filter ->
	      let within x (x1, x2) = x >= x1 && x <= x2 in
	      let filter o = 
		let (x, y) = o#location in
	          within x (x1, x2) && within y (y1, y2) && filter o
	      in
		objects ~filter t @ List.filter filter (Lazy.force near_objs)
	   );
	 method report_damage =
	   (fun obj_damage -> 
	      Client.damage_object t.client obj_damage
	   );
	 method channel = 
	   (fun ch -> new SfxSdl.channel ch)
       end)
  in
(*
    Hashtbl.iter
      (fun _ o -> 
	 o#tick delta (world true o)
      ) t.tanks;

    List.iter (fun o -> o#tick delta (world false o)) t.projectiles;
    t.projectiles <- List.filter (fun o -> o#is_alive) t.projectiles*)

    List.iter
      (fun o ->
	 o#tick delta (world true o)
      ) (objs true);

    List.iter
      (fun o ->
	 o#tick delta (world true o)
      ) (objs false)

      (*  List.iter (fun o -> o#tick delta) (objects t)*)

let center_view at size =
  at -|. size /|. Vector.dup 2.0, size

let handle_io t =
  let now = Unix.gettimeofday () in
    Client.handle_io t.client;
    if now > t.last_sync +. sync_interval then
      begin
	begin
	  match t.pending_ping with
	    | false ->
		t.ping_value := None;
		t.pending_ping <- true;
		ignore (Client.ping t.client t.ping_value)
	    | true ->
		match !(t.ping_value) with
		  | None -> ()
		  | Some latency -> 
		      (*Printf.printf "Latency: %f\n%!" latency;*)
		      t.latency := latency *. 0.1 +. !(t.latency) *. 0.9;
		      Client.report_latency t.client latency;
		      t.pending_ping <- false
	end;
	t.last_sync <- now;
	(* Printf.printf "%d objects\n%!" (List.length (objects ~self_owned:true ~changed_since:t.last_sync_serial t));*)
	let objs () = objects ~self_owned:true ~changed_since:t.last_sync_serial t in
	let remove_ids =
	  List.fold_left
	    (fun del o ->
	       if not o#is_alive then
		 o#get_id::del
	       else
		 del) 
	    []
	    (objs ()) 
	in
	  List.iter 
	    (fun oid ->
	       Client.destruct_object t.client oid
	    ) 
	    remove_ids;
	  List.iter 
	    (fun oid ->
	       Hashtbl.remove t.objects (Object.to_object_id oid)
	    )
	    remove_ids;
	  List.iter 
	    (fun o ->
	       if o#is_alive then
		 Client.report_object t.client (o#get_id, o#get_state 0.0)
	    )
	    (objects ~self_owned:true ~changed_since:t.last_sync_serial t);
	  t.last_sync_serial <- Object.next_serial ()
      end
    (*Printf.printf "%d objects\n%!" (List.length (objects t))*)

let section_of_coords t at =
  let size = 
    (Vector.float2 (t.map.GobaMap.width, t.map.GobaMap.height) +|.
	 (1.0, 1.0)) *|.
	Vector.float2 (t.map.GobaMap.tile_size)
  in
  let section = Vector.int2 (at /|. size *|. (10.0, 10.0)) in
    Printf.sprintf "%c%c" (Char.chr (Char.code 'A' + fst section)) (Char.chr (Char.code '0' + snd section))

let chat t msg =
  Client.chat t.client msg
    
let render t (gfx:#Gfx.t) cur_tanks =
  let disp_base = (10, 10) in

  let objects = objects t in

  let disp_width, disp_height = 640, 400 in

  let base = base t in
  let base_at = base#location in
  let tank_info_bar_width = 50 in
  let tank_display_size = (180.0, 195.0) in
  let main_display_size = (300.0, 
			   float disp_height) in
  let base_info_bar_at = (int_of_float (10.0 +. fst tank_display_size +. float tank_info_bar_width), 
			  int_of_float (snd main_display_size)) in
    (* main view *)
    if try Sys.getenv "GOBA_DEBUG" = "0" with Not_found -> true then
      GobaMap.render_map t.map gfx [`Labels] (disp_base +| (int_of_float (fst tank_display_size +. float tank_info_bar_width) + 10, 
						  0)) 
	(center_view base_at main_display_size) objects;

    let tanks =
      [TankId 0, 
       (tank_info_bar_width, 0), (fun (x, y) -> (x - tank_info_bar_width, y));

       TankId 1, 
       (20 + int_of_float (fst tank_display_size +. fst main_display_size +. float tank_info_bar_width), 0),
       (fun (x, y) -> (x + int_of_float (fst tank_display_size), y));

       TankId 2, 
       (tank_info_bar_width, disp_height - int_of_float (snd tank_display_size)),
       (fun (x, y) -> (x - tank_info_bar_width, y));

       TankId 3, 
       (20 + int_of_float (fst tank_display_size +. fst main_display_size +. float tank_info_bar_width), disp_height - int_of_float (snd tank_display_size)),
       (fun (x, y) -> (x + int_of_float (fst tank_display_size), y));
      ]
    in
      gfx#write ~size:20 (base_info_bar_at +| disp_base) (Printf.sprintf "%d %s" base#get_hp (section_of_coords t base#location));
      List.iter
	(fun (id, at, info_at_f) ->
	   let info_at = info_at_f at +| disp_base in
	     match tank t id with
	       | None -> ()
	       | Some tank ->
		   (*gfx#write ~size:20 (600, text_y) (Vector.f_to_string tank#location);*)
		   GobaMap.render_map t.map gfx [`LabelsExcept [tank#get_id]] (disp_base +| at) 
		     (center_view tank#location tank_display_size) objects;
		   let _ =
		     let damage = tank#get_damage_counter in
		       if damage > 0.0 then
			 let view_alpha, view_bg = 
			   int_of_float (255.0 *. 
					   min 
					   1.0 
					   damage),
			   (255, 0, 0)
			 in
			   gfx#blit (disp_base +| at) (Help.alpha_rectangle gfx#get_surface view_alpha view_bg (Vector.int2 (tank_display_size)));
		   in
		   let bg_color = 
		     if List.exists ((=) id) cur_tanks then
		       (0xad, 0x8a, 0xd6)
		     else
		       (0x7c, 0x5c, 0xa0)
		   in
		   let color = if tank#is_alive then (128, 255, 128) else (64, 128, 64) in
		   let info ?(size=20) at format =
		     Printf.kprintf (fun msg -> gfx#write ~color ~size (info_at +| at) msg) format
		   in
		     gfx#rectangle bg_color info_at (tank_info_bar_width, int_of_float (snd tank_display_size));
		     (*gfx#blit info_at (alpha_rectangle gfx#get_surface 128 (255, 255, 255) (30, 30));*)
		     info ~size:40 (0, 0) "#%d" (match id with TankId i -> i + 1);
		     info (0, 40) "%d" tank#get_hp;
		     info (0, 60) "%s" (section_of_coords t tank#location);
		     info (0, 80) "%d" tank#get_ammo;
		     info (0, 100) "%d" tank#get_mines;

		       ignore (
			 List.fold_left 
			   (fun at (_, surface) ->
			      gfx#blit at surface;
			      at -| (0, 20)
			   )
			   (10, 550)
			   !(t.messages)
		       )
	)
	tanks

