SEXP_CONV_PATH "Goba"

open VecOps

let bitmap = lazy (Bitmaps.bitmap (`Block (11, 18, (128, 128, 0), 255)))
let cmask = lazy (Bitmaps.collision_mask_of_surface (snd (Lazy.force bitmap)))
let bitmap_offset = (-11.0, -18.0) /|. Vector.dup 2.0

type pixel = 
    { mutable at : float * float;
      mutable speed : float * float;
      mutable life_ofs : float;
    }

let explosion_pixels = 20

let explosion_color =
  let red = Vector.float3 (255, 0, 0) in
  let yellow = Vector.float3 (255, 255, 128) in
    fun brightness ->
      (* quantize the value, so we won't generate too many bitmaps (which will be cached) *)
      let r = float (Random.int 10) /. 10.0 in
      let r' = 1.0 -. r in
	Vector.int3 (red *||. Vector.v3 (brightness *. r) +||. yellow *||. Vector.v3 (brightness *. r'))

type init = { init_at : float * float;
	      init_burst : float * float; }
with sexp

class t id init0 =
  let init = Object.default_with init0 init_of_sexp in

object (self)
  inherit Object.t id as super

  val mutable at = init.init_at
  val mutable time_left = 1.0
  val mutable alive = true
  val mutable pixels = 
    Array.of_list (
      GobaUtils.foldl1d 
	(fun _ l ->
	   let ang = Random.float (4.0 *. atan 1.0 *. 2.0) in
	   let dist = 15.0 *. ((Random.float 1.0) ** 1.3) in
	   let ofs = (cos ang *. dist, sin ang *. dist) in
	   let life_ofs = ~-. (Random.float 0.3) in
	   let spd = (ofs +|. init.init_burst) *|. Vector.dup 1.5 in
	     { at = ofs;
	       speed = spd;
	       life_ofs = life_ofs }::l
	)
	[]
	(0, explosion_pixels)
    )
      
  val mutable damage_left = 1.0

  method set_location at' = at <- at'
  method location = at
  method z_location = -.5000.0

  method get_state delay =
    Object.sexp_of_state 
      sexp_of_init
      { Object.state_super = super#get_state delay;
	state_init = init }

  method set_state state delay =
    let state = Object.state_of_sexp init_of_sexp state in
      super#set_state state.Object.state_super delay

  method set_speed (_ : float * float) = ()
  method speed = 0.0, 0.0

  method render (gfx : Gfx.t) _ display_at =
    if self#is_alive then
(*      let ofs, surface = Bitmaps.bitmap (`Block (11, 18, explosion_color 1.0, 255)) in*)
	(*gfx#blit (display_at +| Vector.int2 bitmap_offset +| ofs) surface;*)
	Array.iter 
	  (fun { at = ofs; life_ofs = life_ofs } ->
	     let clamp low high v = min high (max low v) in
	     let left = life_ofs +. time_left in
	       gfx#rectangle ~alpha:(clamp 0 255 (int_of_float (255.0 *. left))) (explosion_color (clamp 0.0 1.0 left)) (display_at +| Vector.int2 ofs) (1, 2))
	  pixels

  method impact world (_:int) = ()

  method collidable = false

  method destructible = false

  method collision_area = 
    ((0.0, 0.0), (0.0, 0.0)), CollisionMask.create_mask 0 0

  method tick delta world =
    time_left <- time_left -. delta;
    let new_alive = time_left > 0.0 in
      if alive <> new_alive then
	self#change;
      alive <- new_alive;
      (*      pixels <- 
	      List.map (fun (at, speed, life_ofs) ->
	      at +|. speed *|. Vector.dup delta, speed, life_ofs) pixels*)
      if alive then
	Array.iteri (fun i { at = at; speed = speed } ->
		       pixels.(i).at <- at +|. speed *|. Vector.dup delta;
		       pixels.(i).speed <- speed *|. Vector.dup (0.5 ** delta);
		    ) pixels;
      if alive && self#self_owned then
	let old_damage = damage_left in
	  damage_left <- damage_left -. delta;
	  let delta = int_of_float old_damage - int_of_float damage_left in
	    if delta > 0 then
	      let radius = 20.0 in
	      let area = ((at -|. Vector.dup radius,
			   at +|. Vector.dup radius)) in
	      let objs = 
		world#objs area (fun obj -> Vector.abs2 (obj#location -|. at) <= radius) in
		let delta = 1 in
		  List.iter (fun o -> o#impact world delta) objs

  method is_alive = alive

  initializer
    (match init0 with 
       | Object.Init _ -> ()
       | Object.State (state0, age) -> self#set_state state0 age);
    self#change;
end
