open VecOps

(* why is slope separately? because we want to calculate it in one
   swoop *)
type mask = { mask_size			: int * int;
	      mask_data			: bool array; }

let create_mask width height =
  { mask_size = width, height;
    mask_data = Array.create (width * height) false;
  }

let rectangle_loc_mask (width, height) =
  (0, 0),
  { mask_size = width, height;
    mask_data = Array.create (width * height) true }

let circle_loc_mask radius =
  let origo = (radius, radius) in
  let size = radius * 2 + 1 in
    origo,
  { mask_size = size, size;
    mask_data = Array.init (size * size)
      (fun ofs ->
	 let at = (ofs / size, ofs mod size) -| origo in
	 let dist2 = fst at * fst at + snd at * snd at in
	   dist2 <= radius * radius
      ) }

let size { mask_size = size } = size

let set_mask mask (x, y) v =
  mask.mask_data.(y * fst mask.mask_size + x) <- v

let get_mask mask (x, y) =
  mask.mask_data.(y * fst mask.mask_size + x)

let or_mask mask (x, y) v =
  mask.mask_data.(y * fst mask.mask_size + x) <- mask.mask_data.(y * fst mask.mask_size + x) || v

let overlap : (int * int) -> (int * int) -> (int * int) option =
  fun (a_x1, a_x2) (b_x1, b_x2) ->
    (*  Printf.printf "Collides? %f,%f with %f,%f\n%!" a1 b1 a2 b2;*)
    if a_x1 < b_x1 then
      begin
	if a_x2 > b_x2 then
	  Some (b_x1, b_x2)
	else if a_x2 > b_x1 then
	  Some (b_x1, a_x2)
	else
	  None
      end
    else
      begin
	if a_x1 >= b_x2 then
	  None
	else if a_x2 > b_x2 then
	  Some (a_x1, b_x2)
	else
	  Some (a_x1, a_x2)
      end

let overlap_2d ((a'x1, a'y1), (a'x2, a'y2)) ((b'x1, b'y1), (b'x2, b'y2)) =
  let ov1 = overlap (a'x1, a'x2) (b'x1, b'x2) in
  let ov2 = overlap (a'y1, a'y2) (b'y1, b'y2) in
    match ov1, ov2 with
      | None, _ -> None
      | _, None -> None
      | Some (x1, x2), Some (y1, y2) ->
	  Some ((x1, y1), (x2, y2))

let blit mask1 (at2, mask2) =
  let at1 = (0, 0) in
  let size1 = mask1.mask_size in
  let size2 = mask2.mask_size in
    begin 
      match overlap_2d (at1, at1 +| size1) (at2, at2 +| size2) with
	| None -> ()
	| Some ((x1, y1), (x2, y2)) -> 
	    ( for x = x1 to x2 - 1 do
		  ( for y = y1 to y2 - 1 do
			let at = (x, y) in
			  or_mask mask1 (at -| at1) (get_mask mask2 (at -| at2))
		    done )
	      done )
    end

let to_string mask =
  let b = Buffer.create 10 in
    Buffer.add_string b (Printf.sprintf "%d\n%d\n" (fst mask.mask_size) (snd mask.mask_size));
    ( for y = 0 to snd mask.mask_size - 1 do
	  ( for x = 0 to fst mask.mask_size - 1 do
		Buffer.add_char b (if get_mask mask (x, y) then 'O' else '.')
	    done );
	Buffer.add_char b '\n'
      done );
    Buffer.contents b

let collides (at1, mask1) (at2, mask2) =
  let size1 = mask1.mask_size in
  let size2 = mask2.mask_size in
    match overlap_2d (at1, at1 +| size1) (at2, at2 +| size2) with
      | None -> false
      | Some ((x1, y1), (x2, y2)) -> 
	  (*Printf.printf "Overlap: %d,%d-%d,%d\n%!" x1 y1 x2 y2;*)
	  let collides = ref false in
	    ( for x = x1 to x2 - 1 do
		  ( for y = y1 to y2 - 1 do
			let at = (x, y) in
			  collides := !collides ||
			    (get_mask mask1 (at -| at1) && get_mask mask2 (at -| at2))
		    done )
	      done );
	    !collides

let window (x, y) n =
  ((x - n, y - n), (x + n + 1, y + n + 1))

let mass_center f region =
  let n, sum =
    GobaUtils.foldl2d 
      (fun at (n, sum) ->
	 if f at then
	   (n + 1, at +| sum)
	 else
	   (n, sum))
      (0, (0, 0))
      region
  in
    if n > 0 then
      Some (Vector.float2 sum *|. Vector.dup (1.0 /. float n))
    else
      None

let denone = function
  | None -> failwith "None not allowed here"
  | Some x -> x

let extreme cond v1 v2 =
  match v1 with
    | None -> Some v2
    | Some x -> 
	if cond v2 x then
	  Some v2
	else
	  v1

let collision_window = 5

let graphics = ref false

let gfx_transform = ref (fun at -> at *| (5, 5))

(*let black = Graphics.black
let red = Graphics.red
let green = Graphics.green
let blue = Graphics.blue
let yellow = Graphics.yellow
let cyan = Graphics.cyan
let magenta = Graphics.magenta

let draw_rect x1 y1 x2 y2 =
  let y1, y2 = 
    if y2 > y1 then
      y1, y2
    else
      y2, y1
  in
    Graphics.draw_rect x1 y1 (x2 - x1 + 1) (y2 - y1 + 1)

let gfx_region color label ((x1, y1), (x2, y2)) =
  if !graphics then 
    begin
      Graphics.set_color color;
      let (x1, y1) = !gfx_transform (x1, y1) in
      let (x2, y2) = !gfx_transform (x2, y2) in
      draw_rect x1 y1 x2 y2;
      let (w, _) = Graphics.text_size label in
	Graphics.moveto ((x1 + x2) / 2 - w / 2) y1;
	Graphics.draw_string label
    end

let gfx_point color label (x, y) =
  if !graphics then 
    begin
      Graphics.set_color color;
      let (x, y) = !gfx_transform (x, y) in
      Graphics.draw_circle x y 2;
      Graphics.moveto x (y + 2);
      Graphics.draw_string label
    end
*)

let collides2 (at1, mask1) (at2, mask2) =
  let size1 = mask1.mask_size in
  let size2 = mask2.mask_size in
(*    gfx_region red "A" (at1, at1 +| size1);
    gfx_region green "B" (at2, at2 +| size2);*)
    match overlap_2d (at1, at1 +| size1) (at2, at2 +| size2) with
      | None -> None
      | Some ab_overlap ->
(*	  gfx_region blue "AB" ab_overlap;*)
	  (*Printf.printf "Overlap: %d,%d-%d,%d\n%!" x1 y1 x2 y2;*)
	  match mass_center (fun at -> get_mask mask1 (at -| at1) && get_mask mask2 (at -| at2)) ab_overlap with
	    | None -> None
	    | Some ab'm ->
(*		gfx_point blue "ab'm" (Vector.int2 ab'm);*)
		match overlap_2d (window (Vector.int2 ab'm) collision_window) (at2, at2 +| size2) with 
		  | None -> failwith "Object collision areas do not overlap at mass center??"
		  | Some window_overlap ->
(*		      gfx_region magenta "Win" window_overlap;*)
		      match mass_center (fun at -> get_mask mask2 (at -| at2)) window_overlap with
			| None -> failwith "Objects do not overlap at mass center??? (overlapping donuts?)"
			| Some b'm ->
			    if !graphics then
			      Printf.printf "ab'm = %f,%f\n"
				(fst ab'm) (snd ab'm);
			    if !graphics then
			      Printf.printf "b'm = %f,%f\n"
				(fst b'm) (snd b'm);
(*			    gfx_point magenta "b'm" (Vector.int2 b'm);*)
			    let uv = Vector.unit (b'm -|. ab'm) in
			      match
				GobaUtils.foldl2d 
				  (fun at (deepest, farthest) -> 
				     if get_mask mask1 (at -| at1) && get_mask mask2 (at -| at2) then
				       let loc = Vector.dot uv (Vector.float2 at) in
					 (extreme (<) deepest loc,
					  extreme (>) farthest loc)
				     else
				       (deepest, farthest)
				  )
				  (None, None)
				  ab_overlap
			      with
				| None, _ | _, None ->
				    failwith "Objects do not overlap at mass center??? (overlapping donuts?) 2"
				| Some deepest, Some farthest ->
				    if !graphics then
				      Printf.printf "deepest = %f, farthest = %f\n%!" 
					deepest farthest;
				    let delta = deepest -. farthest in
				    let fix = Vector.dup (delta) *|. uv in
				      if !graphics then
					Printf.printf "fix = %f, %f\n%!" 
					  (fst fix) (snd fix);
				      Some fix

