SEXP_CONV_PATH "Goba"

type t = { sock		: Unix.file_descr;
	   handle_object: (Protocol.object_info -> float -> unit) ref;
	   handle_destruct: (Object.id -> unit) ref;
	   handle_damage_object : ((Object.id * int) -> unit) ref;
	   handle_chat : (Protocol.chat -> unit) ref;
	   handle_available_players : (Protocol.available_players -> unit) ref;

	   messages_mutex : Mutex.t;
	   error	  : exn option ref;
	   async_messages : (float * Protocol.message') Queue.t;
	   sync_messages  : (float * Protocol.tag * Protocol.message') Queue.t;
	   new_messages	  : Condition.t;

	   handlers	: (Protocol.tag, ((float * Protocol.message') -> unit)) Hashtbl.t;
	 }

let set_object_handler t f =
  t.handle_object := f

let set_destruct_handler t f =
  t.handle_destruct := f

let set_damage_object_handler t f =
  t.handle_damage_object := f

let set_chat_handler t f =
  t.handle_chat := f

let set_available_players_handler t f =
  t.handle_available_players := f

let async_dispatch t delay msg =
  match msg with
    | Protocol.MsgReportObject msg ->
	!(t.handle_object) msg delay
    | Protocol.MsgDestructObject msg ->
	!(t.handle_destruct) msg
    | Protocol.MsgDamageObject msg ->
	!(t.handle_damage_object) msg
    | Protocol.MsgChat msg ->
	!(t.handle_chat) msg
    | Protocol.MsgAvailablePlayers msg ->
	!(t.handle_available_players) msg
    | _ -> raise Protocol.ProtocolError

let receiver_thread fd messages_mutex cond async sync error =
  let receive =
    let buf = String.create 1024 in
      fun () ->
	match GobaUtils.valuefy (Unix.read fd buf 0) (String.length buf) with
	  | `Value 0 -> raise Protocol.ProtocolError
	  | `Value n -> Some (String.sub buf 0 n)
	  | `Exn (Unix.Unix_error (Unix.EAGAIN, _, _)) -> None
	  | `Exn e -> raise e
  in
  let buffer = Buffer.create 1024 in
  let rec loop () =
    match receive () with
      | None -> ()
      | Some str ->
	  let continue =
	    try
	    GobaUtils.fix_buffer buffer str
	      (fun () line ->
		 let msg = Protocol.message_of_sexp (Sexplib.Sexp.of_string line) in
		   `Continue,
	       let now = Unix.gettimeofday () in
		 match msg with
		   | Protocol.Sync tag, msg ->
		       Mutex.lock messages_mutex;
		       Queue.add (now, tag, msg) sync;
		       Mutex.unlock messages_mutex;
		       Condition.broadcast cond
		   | Protocol.Async, msg ->
		       Mutex.lock messages_mutex;
		       Queue.add (now, msg) async;
		       Mutex.unlock messages_mutex;
		       Condition.broadcast cond
	      )
	      ();
	      loop
	    with 
	      | exn -> 
		  error := Some exn;
		  fun () -> ()
	  in
	    continue ()
  in
    loop ()

let nothing _ = ()

let create address =
  let messages_mutex = Mutex.create () in
  let sync_messages = Queue.create () in
  let async_messages = Queue.create () in
  let new_messages = Condition.create () in
  let sock = Net.create_client_socket address in
  let error = ref None in
  let _thread = Thread.create (fun () -> 
				 receiver_thread sock messages_mutex
				   new_messages async_messages sync_messages
				   error
			      ) () in
    { sock = sock;
      handle_object = ref (fun _ _ -> ());
      handle_destruct = ref (fun _ -> ());
      handle_damage_object = ref (fun _ -> ());
      handle_chat = ref (fun _ -> ());
      handle_available_players = ref (fun _ -> ());

      messages_mutex = messages_mutex;
      error = error;
      sync_messages = sync_messages;
      async_messages = async_messages;
      new_messages = new_messages;

      handlers = Hashtbl.create 10;
    }

let array_of_string s =
  Array.init (String.length s) (fun i -> s.[i])

(* assumes mutex is held *)
let check_error t =
  match !(t.error) with
    | None -> ()
    | Some x -> raise x

let wait_sync_message t tag =
  let rec wait () = 
    check_error t;
    if Queue.is_empty t.sync_messages then
      begin
	Condition.wait t.new_messages t.messages_mutex;
	wait ();
      end
    else
      match
	Queue.fold 
	  (fun (found, others) ((time, tag', msg) as msg') ->
	     if tag' = tag then
	       Some msg, others
	     else
	       found, msg'::others
	  )
	  (None, [])
	  t.sync_messages
      with
	| None, _ -> wait ()
	| Some x, others -> 
	    Queue.clear t.sync_messages;
	    List.iter 
	      (fun m -> Queue.add m t.sync_messages)
	      (List.rev others);
	    x
  in
    ThreadUtils.with_mutex t.messages_mutex wait ()

let dump_sync_queue t =
  Printf.printf "Sync queue contents (%d messages):\n" (Queue.length t.sync_messages);
  let now = Unix.gettimeofday () in
  Queue.iter
    (fun (at, tag, msg) ->
       Printf.printf "%f %s\n" 
	 (at -. now)
	 (Protocol.string_of_message (Protocol.Sync tag, msg));
    ) t.sync_messages

let handle_io t =
  let msgs =  
    ThreadUtils.with_mutex t.messages_mutex
      (fun () ->
	 let rec handle msgs =
	   if not (Queue.is_empty t.async_messages) then
	     let v = Queue.pop t.async_messages in
	       handle (v::msgs)
	   else
	     List.rev msgs
	 in
	   check_error t;
	   handle []
      ) ()
  in
    let now = Unix.gettimeofday () in
      List.iter (fun (received, msg) -> async_dispatch t (now -. received) msg) msgs;
      ThreadUtils.with_mutex
	t.messages_mutex
	(fun () -> 
	   let msgs =
	     if (Queue.length t.sync_messages > 3) then
	       dump_sync_queue t;
	     let m = Queue.fold (fun ms el -> el::ms) [] t.sync_messages in
	       List.rev m
	   in
	   let non_handled_msgs =
	     List.rev (
	       List.fold_left
		 (fun handled ((time, tag, msg) as orig_msg) ->
		    match GobaUtils.valuefy (Hashtbl.find t.handlers) tag with
		      | `Value handler ->
			  Hashtbl.remove t.handlers tag;
			  handler (time, msg);
			  handled
		      | `Exn Not_found -> 
			  orig_msg::handled
		      | `Exn e -> raise e
		 ) 
		 []
		 msgs
	     )
	   in
	     Queue.clear t.sync_messages;
	     List.iter (fun msg -> Queue.add msg t.sync_messages) non_handled_msgs)
	()

let command t cmd handler =
  let tag = Protocol.new_tag () in
  let str = Sexplib.Sexp.to_string (Protocol.sexp_of_command 
				      (Protocol.Sync tag, 
				       cmd)) ^ "\n" in
    ignore (Unix.write t.sock str 0 (String.length str));
    Hashtbl.add t.handlers tag handler;
    tag

(*
    let msg = wait_sync_message t in
      (* make sure asynchronous messages are processed before the synchronous *)
      handle_io t;
      msg
*)

let command_wait t cmd =
  let tag = command t cmd (fun _ -> failwith "uh..") in
  let r = wait_sync_message t tag in
    handle_io t;
    r

let async t cmd =
  let str = Sexplib.Sexp.to_string (Protocol.sexp_of_command (Protocol.Async, cmd)) ^ "\n" in
    ignore (Unix.write t.sock str 0 (String.length str))

let handle_failures msg =
  failwith ("Invalid response: " ^ (Sexplib.Sexp.to_string (Protocol.sexp_of_message' msg)))

let report_object t info =
  async t (Protocol.CmdObjectInfo info)

let ping t answer =
  let t0 = Unix.gettimeofday () in
    command t Protocol.CmdPing 
      (function 
	 | t1, Protocol.MsgPong ->
	     let delta = t1 -. t0 in
	       answer := Some delta
	 | _, o -> handle_failures o)
      
let report_latency t latency =
  async t (Protocol.CmdReportLatency latency)

let join t n name =
  match command_wait t (Protocol.CmdJoin (name, Player.Id n)) with
    | Protocol.MsgPlayerJoined j -> Some j
    | Protocol.MsgPlayerJoinFailed -> None
    | o -> handle_failures o

let map t =
  match command_wait t Protocol.CmdMap with
    | Protocol.MsgMap m -> m
    | o -> handle_failures o

let create_object t obj =
  async t (Protocol.CmdCreateObject obj)

let destruct_object t oid =
  async t (Protocol.CmdDestructObject oid)

let damage_object t obj_damage =
  async t (Protocol.CmdDamageObject obj_damage)

let chat t message =
  async t (Protocol.CmdChat message)

let available t =
  match command_wait t Protocol.CmdQueryAvailablePlayers with
    | Protocol.MsgAvailablePlayers m -> m
    | o -> handle_failures o

