open VecOps

type idx =
  [ `Char of (char * Sdlvideo.color)
  | `Block of (int * int * Sdlvideo.color * int)
  | `Bitmap of (string)
  | `Colored of (idx * Sdlvideo.color)
  | `Compose of (idx list)
  | `Move of ((int * int) * idx)
  | `Realpha of (float * idx)
  | `Combine of (idx * idx * idx)
  | `Empty
  ]

type t = { surfaces : (idx, ((int * int) * Sdlvideo.surface)) Hashtbl.t }

let font = lazy (Sdlttf.open_font "font.ttf" 25)

let get_face (ch, color) =
  let font_height = Sdlttf.font_height (Lazy.force font) in
  let s = Sdlttf.render_glyph_blended (Lazy.force font) ch color in
  let (minx, _maxx, _miny, maxy) = Sdlttf.glyph_metrics (Lazy.force font) ch in
    ((minx, font_height - maxy), s)

let create () =
  { surfaces = Hashtbl.create 100; }

let bitmaps = lazy (create ())

let get_bitmaps () = Lazy.force bitmaps

let collision_mask_of_surface s =
  let si = Sdlvideo.surface_info s in
  let module C = CollisionMask in
  let mask = C.create_mask si.Sdlvideo.w si.Sdlvideo.h in
  let set = C.set_mask mask in
    ( for y = 0 to si.Sdlvideo.h - 1 do
	  ( for x = 0 to si.Sdlvideo.w - 1 do
		set (x, y) (Sdlvideo.get_pixel s x y <> 0l)
	    done )
      done );
    mask

let collision_mask_of_surface2 s =
  let si = Sdlvideo.surface_info s in
  let module C = CollisionMask in
  let mask = C.create_mask si.Sdlvideo.w si.Sdlvideo.h in
  let set = C.set_mask mask in
    ( for y = 0 to si.Sdlvideo.h - 1 do
	  ( for x = 0 to si.Sdlvideo.w - 1 do
		set (x, y) (Int32.logand (Sdlvideo.get_pixel s x y) 0x00ffffffl = 0l)
	    done )
      done );
    mask


let map_surface f src =
  let w, h = Gfx.surface_dims src in
  let dst = Sdlvideo.create_RGB_surface_format src [`SRCALPHA] ~w ~h in
    GobaUtils.iter2d 
      (fun (x, y) ->
	 let ((r, g, b), alpha) = Sdlvideo.get_RGBA src (Sdlvideo.get_pixel src x y) in
	   Sdlvideo.put_pixel dst x y 
	     (let (r, g, b, alpha) = 
		f (r, g, b, alpha) in
		Sdlvideo.map_RGB dst ~alpha (r, g, b)
	     )
      )
      ((0, 0), (w, h));
    dst

(* oh why isn't this in the standard library.. *)
let array_for_all f a =
  let len = Array.length a in
  let rec aux idx =
    if idx >= len then
      true
    else
      if f a.(idx) then
	aux (idx + 1)
      else
	false
  in
    aux 0

let map_surfaces f srcs =
  let w, h = Gfx.surface_dims srcs.(0) in
    assert (array_for_all ((=) (w, h)) (Array.map Gfx.surface_dims srcs));
    let srcs_len = Array.length srcs in
    let dst = Sdlvideo.create_RGB_surface_format srcs.(0) [`SRCALPHA] ~w ~h in
    let data = Array.make srcs_len ((0, 0, 0), 0) in
      GobaUtils.iter2d 
	(fun ((x, y) as at) ->
	   for c = 0 to srcs_len - 1 do
	     data.(c) <- Sdlvideo.get_RGBA srcs.(c) (Sdlvideo.get_pixel srcs.(c) x y)
	   done;
	   Sdlvideo.put_pixel dst x y
	     (let (color, alpha) = f at data in
		Sdlvideo.map_RGB dst ~alpha color)
	)
	((0, 0), (w, h));
      dst

(* recolors white/gray *)
let recolor (dr, dg, db) ((r, g, b, a) as orig) =
  let min = List.fold_left min r [g; b] in
  let max = List.fold_left max r [g; b] in
  let delta = max - min in
    if float delta /. float max < 0.05 then
      let br = float (List.fold_left (+) r [g; b]) /. (3.0 *. 255.0) in
      let color = (int_of_float (float dr *. br), 
		   int_of_float (float dg *. br),
		   int_of_float (float db *. br),
		   a)
      in
	color
    else
      orig

let realpha adjust (r, g, b, a) =
  (r, g, b, min 255 (max 0 (adjust a)))

let ipolate_rgba v ((r1, g1, b1), a1) ((r2, g2, b2), a2) =
  let v' = 1.0 -. v in
  let ( *! ) r v = int_of_float (r *. float v) in
    ((v' *! r1 + v *! r2,
      v' *! g1 + v *! g2, 
      v' *! b1 + v *! b2), 
     v' *! a1 + v *! a2)

let get surface x y = Sdlvideo.get_RGBA surface (Sdlvideo.get_pixel surface x y)

(*
let dump_surface s =
  let w, h = Gfx.surface_dims s in
    for y = 0 to h - 1 do
      for x = 0 to w - 1 do
	let color, alpha = get src xc yc in
      done;
      Printf.printf "\n"
    done
*)

(* finally had to resort to: http://en.wikipedia.org/wiki/Alpha_transparency *)
let alpha_blit ~src ~dst ~dstofs =
  let s'w, s'h = Gfx.surface_dims src in
    for xc = 0 to s'w - 1 do
      for yc = 0 to s'h - 1 do
	let dx, dy = xc + fst dstofs, yc + snd dstofs in
	let src_color, src_alpha = get src xc yc in
	let dst_color, dst_alpha = get dst dx dy in
	let f255 v = float v /. 255.0 in
	let new_alpha = int_of_float (min 255.0 (float src_alpha +. float dst_alpha *. (1. -. f255 src_alpha))) in 
	  if new_alpha > 1 then
	    let new_color = 
	      Vector.int3 (Vector.float3 src_color *||. Vector.v3 (float src_alpha /. float new_alpha) +||.
			     Vector.float3 dst_color *||. Vector.v3 ((float dst_alpha *. (1.0 -. f255 src_alpha)) /. float new_alpha)) in
	      Sdlvideo.put_pixel dst dx dy (Sdlvideo.map_RGB dst ~alpha:new_alpha new_color);
(*	      (let r's, g's, b's = src_color in
	       let r'd, g'd, b'd = dst_color in
	       let r, g, b = new_color in
		 Printf.printf "((%d,%d,%d),%d)+((%d,%d,%d),%d)->((%d,%d,%d),%d) " 
		   r's g's b's src_alpha
		   r'd g'd b'd dst_alpha
		   r g b new_alpha)*)
	  else
	    Sdlvideo.put_pixel dst dx dy (Sdlvideo.map_RGB dst ~alpha:0 dst_color);
      done;
(*      Printf.printf "\n";*)
    done

let rec bitmap (idx : idx) =
  let bm = Lazy.force bitmaps in
    try Hashtbl.find bm.surfaces idx
    with Not_found ->
      let (ofs, surface) = 
	match idx with
	  | `Char el -> get_face el
	  | `Block (width, height, color, alpha) ->
	      (0, 0), 
	      (let s = Sdlvideo.create_RGB_surface [`SRCALPHA] ~w:width ~h:height ~bpp:32 ~rmask:0xff0000l ~gmask:0x00ff00l ~bmask:0x0000ffl ~amask:0xff000000l in
		 Sdlvideo.fill_rect s (Sdlvideo.map_RGB ~alpha s color);
		 s)
	  | `Bitmap (file) ->
	      (0, 0),
	      (Sdlloader.load_image (Printf.sprintf "gfx/%s.png" file))
	  | `Colored (idx, color) ->
	      let bm = bitmap idx in
		(fst bm, map_surface (recolor color) (snd bm))
	  | `Compose (idxs) -> compose idxs
	  | `Combine (idxs) -> combine idxs
	  | `Move (ofs, idx) ->
	      let bm = bitmap idx in
		(fst bm +| ofs, snd bm)
	  | `Realpha (adjust, idx) ->
	      let bm = bitmap idx in
		(fst bm, map_surface (realpha (fun a -> int_of_float (float a *. adjust))) (snd bm))
	  | `Empty ->
	      (0, 0), Sdlvideo.create_RGB_surface [] ~w:0 ~h:0 ~bpp:24 ~rmask:0xff0000l ~gmask:0x00ff00l ~bmask:0x0000ffl ~amask:0x0l
		
      in
      let surface = Sdlvideo.display_format ~alpha:true surface in
      let s = (ofs, surface) in
	Hashtbl.add bm.surfaces idx s;
	s

and compose idxs =
  if idxs = [] then
    bitmap `Empty
  else
    let bms = List.map bitmap idxs in
    let extreme f cmp = List.fold_left (fun v b -> if cmp v (f b) then v else (f b)) (f (List.hd bms)) (List.tl bms) in
    let min_x = extreme (fun ((x0, _), _) -> x0) (<) in
    let min_y = extreme (fun ((_, y0), _) -> y0) (<) in
    let max_x = extreme (fun ((x0, _), s) -> x0 + fst (Gfx.surface_dims s)) (>) in
    let max_y = extreme (fun ((_, y0), s) -> y0 + snd (Gfx.surface_dims s)) (>) in
    let ofs = min_x, min_y in
    let size = (max_x, max_y) -| (min_x, min_y) in
    let surface = Sdlvideo.create_RGB_surface [`SRCALPHA] ~w:(fst size) ~h:(snd size) ~bpp:32 ~rmask:0xff0000l ~gmask:0x00ff00l ~bmask:0x0000ffl ~amask:0xff000000l in
      Sdlvideo.fill_rect surface 0x00000000l;
      (*    Printf.printf "Begin composing\n";*)
      List.iter 
	(fun (ofs', bm) ->
	   let ofs = ofs' -| ofs in
	     (*	 let size = Gfx.surface_dims bm in*)
	     (*	   Sdlvideo.blit_surface 
		   ~src:bm
		   ~dst:surface
		   ~dst_rect:(Sdlvideo.rect (fst ofs) (snd ofs) (fst size) (snd size))
		   ()*)
	     alpha_blit ~src:bm ~dst:surface ~dstofs:ofs
	)
	bms;
      (*    Printf.printf "Finish composing\n%!";*)
      ofs, surface

and combine (value, left, right) =
  let idxs = [|value; left; right|] in
  let bms = Array.map bitmap idxs in
  let f _ vs =
    let v = 
      let ((r, g, b), _) = vs.(0) in
	float (r + g + b) /. (255.0 *. 3.0)
    in
      ipolate_rgba v vs.(1) vs.(2)
  in
    assert (array_for_all (fun (ofs, _) -> ofs = fst bms.(0)) bms);
    (fst bms.(0), 
     map_surfaces f (Array.map snd bms))

let dump_stats () =
  Printf.printf "Bitmaps stats: %d surfaces\n" (Hashtbl.length (Lazy.force bitmaps).surfaces)

let _ = at_exit dump_stats
