SEXP_CONV_PATH "Goba"

open VecOps
open GobaUtils

let layout = [|[|0; 0; 1; 1; 0; 0|];
	       [|0; 0; 1; 1; 0; 0|];
	       [|1; 1; 0; 0; 1; 1|];
	       [|1; 1; 0; 0; 1; 1|];
	       [|0; 0; 1; 1; 0; 0|];
	       [|0; 0; 1; 1; 0; 0|]|]
let layout_size = 6, 6
let tile = 11, 18

let base_attack_sound = Sfx.Sample "base-attack"

let recharge_sound = Sfx.Sample "recharge"

let channel = Sfx.alloc_channel ()

type init =
    { init_color : (int * int * int);
      init_at	 : float * float;
    }
with sexp

class t id init0 =
  let init = Object.default_with init0 init_of_sexp in
  let color = init.init_color in
  let bitmap = lazy (Bitmaps.bitmap (`Block (11, 18, color, 255))) in
  let bitmap_offset = Vector.int2 ((-2.5, -2.5) *|. Vector.float2 tile) in
  let cmask = 
    lazy 
      (let module C = CollisionMask in
       let module B = Bitmaps in
       let m = C.create_mask (fst tile * 6) (snd tile * 6) in
       let block_mask = B.collision_mask_of_surface (snd (Lazy.force bitmap)) in
	 ( for xc = 0 to fst layout_size - 1 do
	     ( for yc = 0 to snd layout_size - 1 do
		 if layout.(yc).(xc) = 1 then
		   C.blit m (((xc, yc) *| tile), block_mask)
	       done )
	   done );
	 m
      ) in

object (self : 'a)
  inherit Object.t id as super
  inherit Collision.t

  val mutable at = init.init_at
  val mutable hp = 300
  val mutable next_recharge = 0.0
  val mutable under_attack_sound = false

  method get_state delay =
    Object.sexp_of_state sexp_of_init
      { Object.state_super = super#get_state delay;
	state_init = { init_color = color;
		       init_at = at } }

  method set_state state delay =
    let state = Object.state_of_sexp init_of_sexp state in
      super#set_state state.Object.state_super delay

  method set_location at' = at <- at'

  method location = at
  method set_speed speed' = ()
  method speed = 0.0, 0.0
  method render (gfx:#Gfx.t) (_:Object.render_flags) at = 
    let ofs, surface = Lazy.force bitmap in
    let ofs = ofs +| bitmap_offset in
      ( for xc = 0 to fst layout_size - 1 do
	    ( for yc = 0 to snd layout_size - 1 do
		  if layout.(yc).(xc) = 1 then
		    gfx#blit (at +| ofs +| (xc, yc) *| tile) surface
	      done )
	done )

  method do_recharge (world:Object.world) =
    if self#self_owned then
      let at0 = at +|. Vector.float2 bitmap_offset in
      let size = (6.0, 6.0) *|. Vector.float2 tile in
      let objs = 
	world#objs 
	  (at0, at0 +|. size)
	  (fun _ -> true)
      in
	if List.fold_left (fun recharged o -> o#recharge || recharged) false objs then
	  if not (world#channel channel)#playing then
	    (world#channel channel)#play recharge_sound
	    
  method tick delta (world:Object.world) = 
    if under_attack_sound then
      begin
	under_attack_sound <- false;
	if not (world#channel channel)#playing then
	  (world#channel channel)#play base_attack_sound;
	(world#channel channel)#expire (Some 2.0);
      end;
    next_recharge <- next_recharge -. delta;
    if next_recharge <= 0.0 then
      begin
	next_recharge <- 0.2;
	self#do_recharge world;
      end

  method move f = 
    at <- f at

  method is_alive = hp > 0

  method collision_area = 
    let at0 = at +|. Vector.float2 bitmap_offset in
    let size = (6.0, 6.0) *|. Vector.float2 tile in
      (*Printf.printf "Collision area: %f,%f + %f,%f\n%!" (fst at0) (snd at0) (fst size) (snd size);*)
      (at0, at0 +|. size), Lazy.force cmask

  method get_hp = hp

  method cause_damage damage =
    under_attack_sound <- true;
(*    if not (world.Object.world.world_channel channel)#playing then
    (*	    Sdlmixer.play_channel ~channel:7 (Lazy.force recharge_sound)

      Sdlmixer.play_channel ~loops:~-1 ~channel:7 (Lazy.force base_attack_sound);*)
(*    Sdlmixer.expire_channel 7 (Some 2.0);*)
    (world.Object.world.world_channel channel)#expire (Some 2.0);*)
    hp <- max 0 (hp - damage)

  initializer
    (match init0 with 
       | Object.Init _ -> ()
       | Object.State (state0, age) -> self#set_state state0 age);
    self#change;
end
